'use strict';

const _ = require('lodash');

const secret = process.env.EXPOSED_PRIVATE_API_SECRET;

const privateAuthentication = (req, res, next) => {
  const auth = _.get(req, 'headers.authorization', '').toLowerCase();

  const parts = auth.split(' ').filter(x => x.length);

  if (parts.length !== 2 || parts[0] !== 'bearer' || parts[1] !== secret) {
    return res.sendStatus(403); // Unauthorized
  }

  next();
};
/**
 * This hook will create a private API (secured with a custom secret)
 * You must authenticate with your token to access this API
 * HTTP headers must include: `authorization: bearer YOUR_SECRET_HERE`
 */


const exposeApi = async () => {
  if (!secret || !secret.length) {
    return bp.logger.warn('Custom API disabled because you did not provide an API secret (set the "EXPOSED_PRIVATE_API_SECRET" env variable)');
  }

  const router = bp.http.createRouterForBot('private-api', {
    checkAuthentication: false
  });
  router.use(privateAuthentication);
  /*
    This route exposes a route to create trusted chat references for your webchat visitors
    There are three ways you can use the generated signature:
    1. Standalone Webchat: Append the reference to the URL of the chat, e.g. http://<bot_url>/s/your_bot?ref=[reference]=[signature]
    2. Embedded Webchat: Add the `ref` to the webchat init options. e.g. `{ ref: '[reference]=[signature]', host: ... }`
    3. Send a custom Incoming Event (using the SDK), see the built-in hook "after_incoming_middleware/02_set_session_reference.js"
  */

  router.get('/chat-reference/:reference', async (req, res, next) => {
    try {
      const reference = req.params.reference;
      const signature = await bp.security.getMessageSignature(req.params.reference);
      res.send({
        reference,
        signature,
        full: `${reference}=${signature}`
      });
    } catch (err) {
      bp.logger.attachError(err).error('Error generating signature');
      res.send(500);
    }
  });
  let apiUrl = await router.getPublicPath();
  apiUrl = apiUrl.replace('BOT_ID', '___');
  bp.logger.info(`Private API Path is ${apiUrl}`);
};

return exposeApi();
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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