"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _fs = _interopRequireDefault(require("fs"));

var _moment = _interopRequireDefault(require("moment"));

var _ms = _interopRequireDefault(require("ms"));

var _customAnalytics = _interopRequireDefault(require("./custom-analytics"));

var _stats = _interopRequireDefault(require("./stats"));

var _task = require("./task");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class Analytics {
  constructor(bp, botId) {
    this.bp = bp;
    this.botId = botId;

    _defineProperty(this, "_knex", void 0);

    _defineProperty(this, "_stats", void 0);

    _defineProperty(this, "_task", void 0);

    _defineProperty(this, "custom", void 0);

    this._knex = bp['database'];
    this._stats = new _stats.default(this._knex);
    this._task = new _task.UpdateTask(this.bp, this._getInterval());
    this.custom = (0, _customAnalytics.default)({
      bp,
      botId
    });
  }

  async start() {
    this._task.runTask = async () => {
      await this._updateData();
    };

    await this._task.start(this.botId);
  }

  async stop() {
    await this._task.stop(this.botId);
  }

  async getAnalyticsMetadata() {
    const timestamp = await this._stats.getLastRun();
    const lastRun = (0, _moment.default)(timestamp);

    const elasped = _moment.default.duration((0, _moment.default)().diff(lastRun)).humanize();

    return {
      lastUpdated: elasped,
      size: this._getDBSize()
    };
  }

  async _updateData() {
    const totalUsers = await this._stats.getTotalUsers();
    const activeUsers = await this._stats.getDailyActiveUsers();
    const interactionsRange = await this._stats.getInteractionRanges();
    const avgInteractions = await this._stats.getAverageInteractions();
    const nbUsers = await this._stats.getNumberOfUsers();
    const rentention = await this._stats.usersRetention();
    const busyHours = await this._stats.getBusyHours();
    await this._savePartialData(this.botId, 'analytics', {
      totalUsers: totalUsers || 0,
      activeUsers,
      interactionsRange: interactionsRange,
      fictiveSpecificMetrics: {
        numberOfInteractionInAverage: avgInteractions,
        numberOfUsersToday: nbUsers.today,
        numberOfUsersYesterday: nbUsers.yesterday,
        numberOfUsersThisWeek: nbUsers.week
      },
      retentionHeatMap: rentention,
      busyHoursHeatMap: busyHours
    });
    await this._stats.setLastRun();
  }

  async getChartsGraphData() {
    const analytics = await this.bp.kvs.get(this.botId, 'analytics');

    if (!analytics) {
      return {};
    }

    return {
      loading: false,
      noData: false,
      totalUsersChartData: analytics['totalUsers'] || [],
      activeUsersChartData: analytics['activeUsers'] || [],
      genderUsageChartData: analytics['genderUsage'] || [],
      typicalConversationLengthInADay: analytics['interactionsRange'] || [],
      specificMetricsForLastDays: analytics['fictiveSpecificMetrics'] || {},
      retentionHeatMap: analytics['retentionHeatMap'] || [],
      busyHoursHeatMap: analytics['busyHoursHeatMap'] || []
    };
  }

  _getDBSize() {
    if (this.bp.database.isLite) {
      return _fs.default.statSync(this.bp.database.location)['size'] / 1000000.0; // in megabytes
    } else {
      return 1;
    }
  }

  _getInterval() {
    return this._getDBSize() < 5 ? (0, _ms.default)('5m') : (0, _ms.default)('1h');
  }

  async _savePartialData(botId, property, data) {
    await this.bp.kvs.set(botId, property, data);
  }

}

exports.default = Analytics;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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