"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MessengerClient = exports.MessengerService = void 0;

var _axios = _interopRequireDefault(require("axios"));

var _crypto = _interopRequireDefault(require("crypto"));

var _express = require("express");

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const debug = DEBUG('channel-messenger');
const debugMessages = debug.sub('messages');
const debugHttp = debug.sub('http');
const debugWebhook = debugHttp.sub('webhook');
const debugHttpOut = debugHttp.sub('out');
const outgoingTypes = ['text', 'typing', 'login_prompt', 'carousel'];

class MessengerService {
  constructor(bp) {
    this.bp = bp;

    _defineProperty(this, "http", _axios.default.create({
      baseURL: 'https://graph.facebook.com/v3.2/me'
    }));

    _defineProperty(this, "mountedBots", []);

    _defineProperty(this, "router", void 0);

    _defineProperty(this, "appSecret", void 0);
  }

  async initialize() {
    const config = await this.bp.config.getModuleConfig('channel-messenger');

    if (!config.verifyToken || !config.verifyToken.length) {
      throw new Error('You need to set a non-empty value for "verifyToken" in data/global/config/channel-messenger.json');
    }

    if (!config.appSecret || !config.appSecret.length) {
      throw new Error(`You need to set a non-empty value for "appSecret" in data/global/config/channel-messenger.json`);
    }

    this.appSecret = config.appSecret;
    this.router = this.bp.http.createRouterForBot('channel-messenger', {
      checkAuthentication: false,
      enableJsonBodyParser: false // we use our custom json body parser instead, see below

    });
    this.router.getPublicPath().then(publicPath => {
      if (publicPath.indexOf('https://') !== 0) {
        this.bp.logger.warn('Messenger requires HTTPS to be setup to work properly. See EXTERNAL_URL botpress config.');
      }

      this.bp.logger.info(`Messenger Webhook URL is ${publicPath.replace('BOT_ID', '___')}/webhook`);
    });
    this.router.use((0, _express.json)({
      verify: this._verifySignature.bind(this)
    }));
    this.router.get('/webhook', this._setupWebhook.bind(this));
    this.router.post('/webhook', this._handleIncomingMessage.bind(this));
    this.bp.events.registerMiddleware({
      description: 'Sends outgoing messages for the messenger channel',
      direction: 'outgoing',
      handler: this._handleOutgoingEvent.bind(this),
      name: 'messenger.sendMessages',
      order: 200
    });
  }

  async mountBot(botId) {
    const config = await this.bp.config.getModuleConfigForBot('channel-messenger', botId);

    if (config.enabled) {
      if (!config.accessToken) {
        return this.bp.logger.forBot(botId).error('You need to configure an Access Token to enable it. Messenger Channel is disabled for this bot.');
      }

      const {
        data
      } = await this.http.get('/', {
        params: {
          access_token: config.accessToken
        }
      });

      if (!data || !data.id) {
        return this.bp.logger.forBot(botId).error('Could not register bot, are you sure your Access Token is valid? Messenger Channel is disabled for this bot.');
      }

      const pageId = data.id;
      const client = new MessengerClient(botId, this.bp, this.http);
      this.mountedBots.push({
        botId: botId,
        client,
        pageId
      });
      await client.setupGreeting();
      await client.setupGetStarted();
      await client.setupPersistentMenu();
    }
  }

  async unmountBot(botId) {
    this.mountedBots = _lodash.default.remove(this.mountedBots, x => x.botId === botId);
  }

  getMessengerClientByBotId(botId) {
    const entry = _lodash.default.find(this.mountedBots, x => x.botId === botId);

    if (!entry) {
      throw new Error(`Can't find a MessengerClient for bot "${botId}"`);
    }

    return entry.client;
  } // See: https://developers.facebook.com/docs/messenger-platform/webhook#security


  _verifySignature(req, res, buffer) {
    const signatureError = new Error("Couldn't validate the request signature.");

    if (!/^\/webhook/i.test(req.path)) {
      return;
    }

    const signature = req.headers['x-hub-signature'];

    if (!signature || !this.appSecret) {
      throw signatureError;
    } else {
      const [, hash] = signature.split('=');

      const expectedHash = _crypto.default.createHmac('sha1', this.appSecret).update(buffer).digest('hex');

      if (hash !== expectedHash) {
        debugWebhook('invalid signature', req.path);
        throw signatureError;
      } else {
        debugWebhook('signed', req.path);
      }
    }
  }

  async _handleIncomingMessage(req, res) {
    const body = req.body;

    if (body.object !== 'page') {// TODO: Handle other cases here
    } else {
      res.status(200).send('EVENT_RECEIVED');
    }

    for (const entry of body.entry) {
      const pageId = entry.id;
      const messages = entry.messaging;

      const bot = _lodash.default.find(this.mountedBots, {
        pageId
      });

      if (!bot) {
        debugMessages('could not find a bot for page id =', pageId);
        continue;
      }

      for (const webhookEvent of messages) {
        if (!webhookEvent.sender) {
          return;
        }

        debugMessages('incoming', webhookEvent);
        const senderId = webhookEvent.sender.id;
        await bot.client.sendAction(senderId, 'mark_seen');

        if (webhookEvent.message) {
          await this._sendEvent(bot.botId, senderId, webhookEvent.message, {
            type: 'message'
          });
        } else if (webhookEvent.postback) {
          await this._sendEvent(bot.botId, senderId, {
            text: webhookEvent.postback.payload
          }, {
            type: 'callback'
          });
        }
      }
    }
  }

  async _sendEvent(botId, senderId, message, args) {
    this.bp.events.sendEvent(this.bp.IO.Event({
      botId,
      channel: 'messenger',
      direction: 'incoming',
      payload: message,
      preview: message.text,
      target: senderId,
      ...args
    }));
  }

  async _setupWebhook(req, res) {
    const mode = req.query['hub.mode'];
    const token = req.query['hub.verify_token'];
    const challenge = req.query['hub.challenge'];
    const config = await this.bp.config.getModuleConfig('channel-messenger');

    if (mode && token && mode === 'subscribe' && token === config.verifyToken) {
      this.bp.logger.debug('Webhook Verified');
      res.status(200).send(challenge);
    } else {
      res.sendStatus(403);
    }
  }

  async _handleOutgoingEvent(event, next) {
    if (event.channel !== 'messenger') {
      return next();
    }

    const messageType = event.type === 'default' ? 'text' : event.type;
    const messenger = this.getMessengerClientByBotId(event.botId);

    if (!_lodash.default.includes(outgoingTypes, messageType)) {
      return next(new Error('Unsupported event type: ' + event.type));
    }

    if (messageType === 'typing') {
      const typing = parseTyping(event.payload.value);
      await messenger.sendAction(event.target, 'typing_on');
      await Promise.delay(typing);
      await messenger.sendAction(event.target, 'typing_off');
    } else if (messageType === 'text' || messageType === 'carousel') {
      await messenger.sendTextMessage(event.target, event.payload);
    } else {
      // TODO We don't support sending files, location requests (and probably more) yet
      throw new Error(`Message type "${messageType}" not implemented yet`);
    }

    next(undefined, false);
  }

}

exports.MessengerService = MessengerService;

class MessengerClient {
  constructor(botId, bp, http) {
    this.botId = botId;
    this.bp = bp;
    this.http = http;

    _defineProperty(this, "config", void 0);
  }

  async getConfig() {
    if (this.config) {
      return this.config;
    }

    const config = await this.bp.config.getModuleConfigForBot('channel-messenger', this.botId);

    if (!config) {
      throw new Error(`Could not find channel-messenger.json config file for ${this.botId}.`);
    }

    return config;
  }

  async setupGetStarted() {
    const config = await this.getConfig();

    if (!config.getStarted) {
      return;
    }

    await this.sendProfile({
      get_started: {
        payload: config.getStarted
      }
    });
  }

  async setupGreeting() {
    const config = await this.getConfig();

    if (!config.greeting) {
      await this.deleteProfileFields(['greeting']);
      return;
    }

    await this.sendProfile({
      greeting: [{
        locale: 'default',
        text: config.greeting
      }]
    });
  }

  async setupPersistentMenu() {
    const config = await this.getConfig();

    if (!config.persistentMenu || !config.persistentMenu.length) {
      await this.deleteProfileFields(['persistent_menu']);
      return;
    }

    await this.sendProfile({
      persistent_menu: config.persistentMenu
    });
  }

  async sendAction(senderId, action) {
    const body = {
      recipient: {
        id: senderId
      },
      sender_action: action
    };
    debugMessages('outgoing action', {
      senderId,
      action,
      body
    });
    await this._callEndpoint('/messages', body);
  }

  async sendTextMessage(senderId, message) {
    const body = {
      recipient: {
        id: senderId
      },
      message
    };
    debugMessages('outgoing text message', {
      senderId,
      message,
      body
    });
    await this._callEndpoint('/messages', body);
  }

  async deleteProfileFields(fields) {
    const endpoint = '/messenger_profile';
    const config = await this.getConfig();
    debugHttpOut(endpoint, fields);
    await this.http.delete(endpoint, {
      params: {
        access_token: config.accessToken
      },
      data: {
        fields
      }
    });
  }

  async sendProfile(message) {
    await this._callEndpoint('/messenger_profile', message);
  }

  async _callEndpoint(endpoint, body) {
    const config = await this.getConfig();
    debugHttpOut(endpoint, body);
    await this.http.post(endpoint, body, {
      params: {
        access_token: config.accessToken
      }
    });
  }

}

exports.MessengerClient = MessengerClient;

function parseTyping(typing) {
  if (isNaN(typing)) {
    return 1000;
  }

  return Math.max(typing, 500);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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