"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _fs = require("fs");

var _lodash = _interopRequireDefault(require("lodash"));

var _tmp = _interopRequireDefault(require("tmp"));

var _verror = require("verror");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const debug = DEBUG('nlu').sub('intents');
const debugTrain = debug.sub('train');
const debugPredict = debug.sub('predict');

class FastTextClassifier {
  constructor(toolkit, logger, ftOverrides) {
    this.toolkit = toolkit;
    this.logger = logger;
    this.ftOverrides = ftOverrides;

    _defineProperty(this, "_modelsByContext", {});

    _defineProperty(this, "prebuiltWordVecPath", void 0);
  }

  getFastTextParams() {
    const extraArgs = this.prebuiltWordVecPath ? {
      pretrainedVectors: this.prebuiltWordVecPath
    } : {};
    return { ...extraArgs,
      lr: _lodash.default.get(this.ftOverrides, 'learningRate', 0.8),
      epoch: _lodash.default.get(this.ftOverrides, 'epoch', 5),
      wordNgrams: _lodash.default.get(this.ftOverrides, 'wordNgrams', 3)
    };
  }

  sanitizeText(text) {
    return text.toLowerCase().replace(/\t|\r|\f/gi, ' ').replace(/\s\s+/gi, ' ');
  }

  _writeTrainingSet(intents, trainingFilePath) {
    const fileStream = (0, _fs.createWriteStream)(trainingFilePath, {
      flags: 'a'
    });

    for (const intent of intents) {
      intent.utterances.forEach(text => {
        const clean = this.sanitizeText(text);
        fileStream.write(`__label__${intent.name} ${clean}\n`);
      });
    }

    return Promise.fromCallback(cb => fileStream.end(cb));
  }

  teardownModels() {
    if (this._modelsByContext) {
      _lodash.default.values(this._modelsByContext).forEach(x => x.cleanup());
    }
  }

  _hasSufficientData(intents) {
    const datasetSize = _lodash.default.flatMap(intents, intent => intent.utterances).length;

    return intents.length > 0 && datasetSize > 0;
  }

  async _trainForOneModel(intents, modelName) {
    const dataFn = _tmp.default.tmpNameSync({
      prefix: modelName + '-',
      postfix: '.txt'
    });

    await this._writeTrainingSet(intents, dataFn);

    const modelFn = _tmp.default.tmpNameSync({
      postfix: '.bin'
    }); // TODO Apply parameters from Grid-search here


    const ft = new this.toolkit.FastText.Model();
    const params = { ...this.getFastTextParams(),
      input: dataFn
    };
    debugTrain('training fastText model', {
      modelName,
      fastTextParams: params
    });
    await ft.trainToFile('supervised', modelFn, params);
    debugTrain('done with fastText model');
    return {
      ft,
      data: (0, _fs.readFileSync)(modelFn)
    };
  }

  async train(intents) {
    const contextNames = _lodash.default.uniq(_lodash.default.flatMap(intents, x => x.contexts));

    const models = [];
    const modelsByContext = {};
    debugTrain('contexts', contextNames);

    for (const context of contextNames) {
      // TODO Make the `none` intent undeletable, mandatory and pre-filled with random data
      const intentSet = intents.filter(x => x.contexts.includes(context) || x.name === 'none');

      if (this._hasSufficientData(intentSet)) {
        debugTrain('training context', context);

        try {
          const {
            ft,
            data
          } = await this._trainForOneModel(intentSet, context);
          modelsByContext[context] = ft;
          models.push({
            name: context,
            model: data
          });
        } catch (err) {
          throw new _verror.VError(err, `Error training set of intents for context "${context}"`);
        }
      } else {
        debugTrain('insufficent data, skip training context', context);
      }
    }

    this.teardownModels();
    this._modelsByContext = modelsByContext;
    return models;
  }

  async load(models) {
    const m = {};

    if (!models.length) {
      throw new Error(`You need to provide at least one model to load`);
    }

    if (_lodash.default.uniqBy(models, 'name').length !== models.length) {
      const names = models.map(x => x.name).join(', ');
      throw new Error(`You can't train different models with the same names. Names = [${names}]`);
    }

    for (const model of models) {
      const tmpFn = _tmp.default.tmpNameSync({
        postfix: '.bin',
        prefix: model.name + '-'
      });

      (0, _fs.writeFileSync)(tmpFn, model.model);
      const ft = new this.toolkit.FastText.Model();
      await ft.loadFromFile(tmpFn);
      m[model.name] = ft;
    }

    this.teardownModels();
    this._modelsByContext = m;
  }

  async _predictForOneModel(sanitizedInput, modelName) {
    if (!this._modelsByContext[modelName]) {
      throw new Error(`Can't predict for model named "${modelName}" (model not loaded)`);
    }

    try {
      const pred = await this._modelsByContext[modelName].predict(sanitizedInput, 5);

      if (!pred || !pred.length) {
        return [];
      }

      return pred.map(x => ({
        name: x.label.replace('__label__', ''),
        confidence: x.value,
        context: modelName
      }));
    } catch (e) {
      throw new _verror.VError(e, `Error predicting intent for model "${modelName}"`);
    }
  }

  async predict(input, includedContexts = []) {
    if (!Object.keys(this._modelsByContext).length) {
      throw new Error('No model loaded. Make sure you `load` your models before you call `predict`.');
    }

    const sanitized = this.sanitizeText(input); // TODO change this context discriminatin by a weighted scoring instead
    // Add weights and affect the confidence results accordingly
    // ** no scoring algorithm has been choosen, impl is yet to be done

    const modelNames = Object.keys(this._modelsByContext).filter(ctx => !includedContexts.length || includedContexts.includes(ctx));

    try {
      debugPredict('prediction request %o', {
        includedContexts,
        input,
        sanitized
      });
      const predictions = await Promise.map(modelNames, modelName => this._predictForOneModel(sanitized, modelName));
      debugPredict('predictions done %o', {
        includedContexts,
        input,
        sanitized,
        predictions
      });
      return _lodash.default.chain(predictions).flatten().orderBy('confidence', 'desc').uniqBy(x => x.name).value();
    } catch (e) {
      throw new _verror.VError(e, `Error predicting intent for "${sanitized}"`);
    }
  }

}

exports.default = FastTextClassifier;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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