"use strict";

const base = require('./_base');

function render(data) {
  const events = [];

  if (data.typing) {
    events.push({
      type: 'typing',
      value: data.typing
    });
  }

  return [...events, {
    text: data.text,
    quick_replies: data.choices.map(c => ({
      title: c.title,
      payload: c.value.toUpperCase()
    })),
    typing: data.typing
  }];
}

function renderMessenger(data) {
  return [{
    type: 'typing',
    value: data.typing
  }, {
    text: data.text,
    quick_replies: data.choices.map(c => ({
      content_type: 'text',
      title: c.title,
      payload: c.value.toUpperCase()
    }))
  }];
}

function renderElement(data, channel) {
  if (channel === 'web' || channel === 'api' || channel === 'telegram') {
    return render(data);
  } else if (channel === 'messenger') {
    return renderMessenger(data);
  }

  return []; // TODO Handle channel not supported
}

module.exports = {
  id: 'builtin_single-choice',
  group: 'Built-in Messages',
  title: 'Single Choice',
  jsonSchema: {
    description: 'Suggest choices to the user with the intention of picking only one (with an optional message)',
    type: 'object',
    required: ['choices'],
    properties: {
      text: {
        type: 'string',
        title: 'Message'
      },
      choices: {
        type: 'array',
        title: 'Choices',
        minItems: 1,
        maxItems: 10,
        items: {
          type: 'object',
          required: ['title', 'value'],
          properties: {
            title: {
              description: 'The title of the choice (this is what gets shown to the user)',
              type: 'string',
              title: 'Message'
            },
            value: {
              description: 'The value that your bot gets when the user picks this choice (usually hidden from the user)',
              type: 'string',
              title: 'Value'
            }
          }
        }
      },
      ...base.typingIndicators
    }
  },
  uiSchema: {
    text: {
      'ui:field': 'i18n_field'
    },
    choices: {
      'ui:field': 'i18n_array'
    }
  },
  computePreviewText: formData => formData.choices && formData.text && `Choices (${formData.choices.length}) ${formData.text}`,
  renderElement: renderElement,
  hidden: true
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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