"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _moment = _interopRequireDefault(require("moment"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class Stats {
  constructor(knex) {
    this.knex = knex;
  }

  rangeDates() {
    return this.knex('srv_channel_users').select(this.knex.raw('max(created_at) as max, min(created_at) as min')).then().get(0).then(result => {
      if (!result.min || !result.max) {
        return undefined;
      }

      const range = (0, _moment.default)(result.max).diff((0, _moment.default)(result.min), 'days');
      const ranges = [];

      for (let i = 1; i <= 10; i++) {
        ranges.push(parseInt(result.min + range / 10 * i));
      }

      const ret = {
        min: result.min,
        max: result.max,
        format: undefined,
        ranges: ranges
      };

      if (range < 360) {
        ret.format = date => (0, _moment.default)(date).format('MMM Do');
      } else {
        // > 1year period
        ret.format = date => (0, _moment.default)(date).format('MMM YY');
      }

      return ret;
    });
  }

  getTotalUsers() {
    return this.rangeDates().then(dates => {
      if (!dates) {
        return;
      }

      return this.knex('srv_channel_users').select(this.knex.raw('distinct channel')).then(channels => {
        const statsBase = channels.reduce((acc, curr) => {
          acc[curr.channel] = 0;
          return acc;
        }, {
          total: 0
        });
        return this.knex('srv_channel_users').select(this.knex.raw('count(*) as count, max(created_at) as date, channel')).groupBy(this.knex.raw('date(created_at), channel')).orderBy(this.knex.raw('date(created_at)')).then(rows => {
          let total = 0;
          const totalChannel = {};
          const result = {};
          const min = dates.format((0, _moment.default)(new Date(dates.min)).subtract(1, 'day'));
          result[min] = Object.assign({}, statsBase);
          rows.map(row => {
            const date = dates.format(row.date);

            if (!result[date]) {
              result[date] = Object.assign({}, statsBase);
            }

            if (!totalChannel[row.channel]) {
              totalChannel[row.channel] = 0;
            }

            const count = parseInt(row.count);
            totalChannel[row.channel] += count;
            result[date].total = total += count;
            result[date][row.channel] = totalChannel[row.channel];
          });
          const max = dates.format((0, _moment.default)(new Date(dates.max)).add(1, 'hour'));
          result[max] = Object.assign({}, statsBase, {
            total: total
          }, totalChannel);
          return _lodash.default.toPairs(result).map(([k, v]) => {
            v['name'] = k;
            return v;
          });
        });
      });
    });
  }

  getLastDaysRange(nb) {
    const nbOfDays = nb || 14;

    const ranges = _lodash.default.times(nbOfDays, Number);

    return ranges.map(n => {
      const date = (0, _moment.default)(new Date()).subtract(n, 'days');
      return {
        date: date.format('MMM Do'),
        start: date.startOf('day').toDate(),
        end: date.endOf('day').toDate(),
        day: date.format('l')
      };
    });
  }

  getDailyActiveUsers() {
    const ranges = _lodash.default.reverse(this.getLastDaysRange());

    return Promise.mapSeries(ranges, range => {
      return this.knex('analytics_interactions as ai').select(this.knex.raw('count(*) as count, ai.channel')).join('srv_channel_users', 'srv_channel_users.user_id', 'ai.user_id').where(this.knex.date.isBetween('ts', range['start'], range['end'])).where('direction', '=', 'in').groupBy(['ai.user_id', 'ai.channel']).then(results => {
        return results.reduce(function (acc, curr) {
          const count = parseInt(curr.count);
          acc.total += count;
          acc[curr.channel] = count;
          return acc;
        }, {
          total: 0,
          name: range['date']
        });
      });
    });
  } // FIXME: Fix or remove, gender is not a valid column anymore
  // getDailyGender() {
  //   const ranges = _.reverse(this.getLastDaysRange())
  //   return Promise.mapSeries(ranges, range => {
  //     return this.knex('analytics_interactions')
  //       .select(this.knex.raw('count(*) as count, gender'))
  //       .join('users', 'users.id', 'analytics_interactions.user')
  //       .where(this.knex.date.isBetween('ts', range.start, range.end))
  //       .andWhere('direction', '=', 'in')
  //       .groupBy(['user', 'gender'])
  //       .then(results => {
  //         return results.reduce(
  //           function(acc, curr) {
  //             const count = parseInt(curr.count)
  //             acc.total += count
  //             acc[curr.gender] = count
  //             return acc
  //           },
  //           { total: 0, name: range.date }
  //         )
  //       })
  //   })
  // }


  getInteractionRanges() {
    const ranges = this.getLastDaysRange();
    return Promise.mapSeries(ranges, range => {
      const inner = this.knex.from('analytics_interactions').where(this.knex.date.isBetween('ts', range['start'], range['end'])).andWhere('direction', '=', 'in').groupBy('user_id').select(this.knex.raw('count(*) as c')).toString();
      return this.knex.raw(`select
        sum(r1) as s1,
        sum(r2) as s2,
        sum(r3) as s3,
        sum(r4) as s4,
        sum(r5) as s5,
        sum(r6) as s6,
        sum(r7) as s7,
        sum(r8) as s8
      from (select
        (select count(*) as count where c between 0 and 1) as r1,
        (select count(*) where c between 2 and 3) as r2,
        (select count(*) where c between 4 and 5) as r3,
        (select count(*) where c between 6 and 9) as r4,
        (select count(*) where c between 10 and 14) as r5,
        (select count(*) where c between 15 and 29) as r6,
        (select count(*) where c between 30 and 50) as r7,
        (select count(*) where c > 50) as r8
          from (` + inner + `) as q1 ) as q2`);
    }).then(rows => {
      if (rows[0].rows) {
        return rows.map(r => r.rows[0]);
      } else {
        return rows.map(r => r[0]);
      }
    }).then(results => {
      return results.reduce(function (acc, curr) {
        return _lodash.default.mapValues(acc, (a, k) => {
          return a + (parseInt(curr[k]) || 0);
        });
      }, {
        s1: 0,
        s2: 0,
        s3: 0,
        s4: 0,
        s5: 0,
        s6: 0,
        s7: 0,
        s8: 0
      });
    }).then(results => {
      return [{
        name: '[0-2]',
        count: results.s1
      }, {
        name: '[2-4]',
        count: results.s2
      }, {
        name: '[4-6]',
        count: results.s3
      }, {
        name: '[6-10]',
        count: results.s4
      }, {
        name: '[10-15]',
        count: results.s5
      }, {
        name: '[15-30]',
        count: results.s6
      }, {
        name: '[30-50]',
        count: results.s7
      }, {
        name: '50+',
        count: results.s8
      }];
    });
  }

  getAverageInteractions() {
    // Average incoming interactions per user per day for the last 7 days
    const lastWeek = (0, _moment.default)(new Date()).subtract(7, 'days').toDate();
    const now = this.knex.date.now();
    const knx = this.knex;
    return this.knex.select(this.knex.raw('avg(c) as count')).from(function () {
      return this.from('analytics_interactions').where(knx.date.isBetween('ts', lastWeek, now)).andWhere('direction', '=', 'in').groupBy(knx.raw('user_id, date(ts)')).select(knx.raw('count(*) as c')).as('q1');
    }).then().get(0).then(result => {
      return parseFloat(result.count) || 0.0;
    });
  }

  getNumberOfUsers() {
    const knx = this.knex; // Get total number of active users for today, yesterday, this week

    const ranges = [{
      label: 'today',
      start: (0, _moment.default)(new Date()).startOf('day').toDate(),
      end: new Date()
    }, {
      label: 'yesterday',
      start: (0, _moment.default)(new Date()).subtract(1, 'days').startOf('day').toDate(),
      end: (0, _moment.default)(new Date()).subtract(1, 'days').endOf('day').toDate()
    }, {
      label: 'week',
      start: (0, _moment.default)(new Date()).startOf('week').toDate(),
      end: (0, _moment.default)(new Date()).endOf('week').toDate()
    }];
    return Promise.mapSeries(ranges, range => {
      return this.knex.select(this.knex.raw('count(*) as count')).from(function () {
        return this.from('analytics_interactions').where(knx.date.isBetween('ts', range['start'], range['end'])).andWhere('direction', '=', 'in').groupBy('user_id').select(knx.raw(1)).as('q1');
      }).then().get(0).then(result => ({
        label: range['label'],
        count: result.count
      }));
    }).then(results => {
      return results.reduce((acc, curr) => {
        acc[curr.label] = curr.count;
        return acc;
      }, {});
    });
  }

  usersRetention() {
    const knx = this.knex; // Get the last 7 days cohort of users along with the retention rate

    let cohorts = _lodash.default.times(8, n => Number(8 - n));

    cohorts = cohorts.map(n => {
      const day = (0, _moment.default)().subtract(n, 'days');
      return {
        start: day.startOf('day').toDate(),
        end: day.endOf('day').toDate(),
        name: day.format('MMM Do'),
        date: day
      };
    });
    const result = {}; // For each days of the cohort

    return Promise.mapSeries(cohorts, coo => {
      const cohortStart = _lodash.default.get(coo, 'start');

      const cohortEnd = _lodash.default.get(coo, 'end');

      const cohortName = _lodash.default.get(coo, 'name'); // Compute the cohort size [i.e. how many new users on this day?]


      return this.knex('srv_channel_users').where(this.knex.date.isBetween('created_at', cohortStart, cohortEnd)).select(this.knex.raw('count(*) as cohort_size')).then().get(0).then(({
        cohort_size
      }) => {
        cohort_size = parseFloat(cohort_size); // Compute the next 7 days of the cohort
        // and check how many users [from this cohort] spoke on or before this date
        // A user is considered as retentioned if he interacted with the bot any day after he onboarded

        const daysToAdd = _lodash.default.times(7, n => n); // from 0 to 6


        return Promise.mapSeries(daysToAdd, dta => {
          const since = (0, _moment.default)(cohortStart).add(dta, 'days').endOf('day').toDate(); // +x days

          return this.knex.from(function () {
            this.from('analytics_interactions').join('srv_channel_users', 'analytics_interactions.user_id', 'srv_channel_users.user_id') // where he is a member a this cohort
            .where(knx.date.isBetween('created_at', cohortStart, cohortEnd)) // and where he interacted with the bot since onboard+X days
            .andWhere(knx.date.isAfter('ts', since)) // and where the user spoke, not the bot
            .andWhere('direction', '=', 'in').groupBy('srv_channel_users.user_id') // returns the number of interactions per user
            .select(knx.raw('count(*) as interaction_count')).as('q1');
          }).select(this.knex.raw('count(*) as partial_retention')) // return the total number of users
          .then().get(0).then(({
            partial_retention
          }) => {
            partial_retention = parseFloat(partial_retention); // if the date is out of the cohort sample

            if ((0, _moment.default)(since).startOf('day').isSameOrAfter((0, _moment.default)().startOf('day'))) {
              return undefined;
            }

            return partial_retention / cohort_size || 0;
          });
        }).then(retention => {
          const mean = _lodash.default.mean(_lodash.default.filter(retention, v => v !== undefined));

          result[cohortName] = [cohort_size, ...retention, mean];
        });
      });
    }).then(() => result);
  }

  getBusyHours() {
    const ranges = this.getLastDaysRange(7);
    const result = {};
    return Promise.mapSeries(ranges, range => {
      // select count(*) as count, ts from interactions
      // group by strftime('%H', ts/1000, 'unixepoch')
      return this.knex('analytics_interactions').where(this.knex.date.isBetween('ts', range.start, range.end)).select(this.knex.raw('count(*) as count, ' + this.knex.date.hourOfDay('ts').sql + ' as ts')).groupBy(this.knex.date.hourOfDay('ts')).then(rows => {
        const row = [];

        _lodash.default.times(24, () => row.push(0));

        const biggest = rows.reduce((acc, curr) => {
          return acc = curr.count > acc ? curr.count : acc;
        }, 0);
        rows.map(x => {
          row[parseInt(x.ts)] = Math.min(Number((x.count / biggest).toFixed(2)), 0.75);
        });
        result[range.date] = row;
      });
    }).then(() => result);
  }

  getLastRun() {
    return this.knex('analytics_runs').orderBy('ts', 'desc').limit(1).then().get(0).then(entry => {
      return entry && (0, _moment.default)(entry.ts);
    });
  }

  async setLastRun() {
    return this.knex('analytics_runs').insert({
      ts: this.knex.date.now()
    });
  }

}

exports.default = Stats;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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