"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.NLU_PREFIX = void 0;

var _axios = _interopRequireDefault(require("axios"));

var _lodash = _interopRequireDefault(require("lodash"));

var _generate = _interopRequireDefault(require("nanoid/generate"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const safeId = (length = 10) => (0, _generate.default)('1234567890abcdefghijklmnopqrsuvwxyz', length);

const slugify = s => (s || '').toLowerCase().replace(/[^a-z0-9]/g, '_');

const getQuestionId = ({
  questions
}) => `${safeId()}_${slugify(questions[0]).replace(/^_+/, '').substring(0, 50).replace(/_+$/, '')}`;

const NLU_PREFIX = '__qna__';
exports.NLU_PREFIX = NLU_PREFIX;

const getIntentId = id => `${NLU_PREFIX}${id}`;

const normalizeQuestions = questions => questions.map(q => q.replace(/[\r\n]+/g, ' ').replace(/\s+/g, ' ').trim()).filter(Boolean);

class Storage {
  constructor(bp, config, botId) {
    _defineProperty(this, "bp", void 0);

    _defineProperty(this, "config", void 0);

    _defineProperty(this, "botId", void 0);

    _defineProperty(this, "categories", void 0);

    this.bp = bp;
    this.config = config;
    this.botId = botId;

    if (config.qnaCategories && config.qnaCategories.length > 0) {
      this.categories = config.qnaCategories.split(',').map(x => x.trim()).filter(x => x.length);
    }
  }

  async getAxiosConfig() {
    return this.bp.http.getAxiosConfigForBot(this.botId, {
      localUrl: true
    });
  }

  async initialize() {
    await this.syncQnaToNlu();
  } // TODO Find better way to implement. When manually copying QNA, intents are not created.
  // Manual edit & save of each one is required for the intent to be created.


  async syncQnaToNlu() {
    const axiosConfig = await this.getAxiosConfig();
    const allQuestions = await this.fetchAllQuestions();
    const {
      data: allIntents
    } = await _axios.default.get(`/mod/nlu/intents`, axiosConfig);

    for (const question of allQuestions) {
      const matchedIntent = _lodash.default.find(allIntents, intent => intent.name === getIntentId(question.id).toLowerCase());

      if (question.data.enabled && !matchedIntent) {
        const intent = {
          name: getIntentId(question.id),
          entities: [],
          contexts: [question.data.category || 'global'],
          utterances: normalizeQuestions(question.data.questions)
        };
        await _axios.default.post(`/mod/nlu/intents`, intent, axiosConfig);
        this.bp.logger.info(`Created NLU intent for QNA ${question.id}`);
      } else if (!question.data.enabled && matchedIntent) {
        await this.bp.ghost.forBot(this.botId).deleteFile(this.config.qnaDir, `${question.id}.json`);
      }
    }
  }

  async update(data, id) {
    const axiosConfig = await this.getAxiosConfig();
    id = id || getQuestionId(data);

    if (data.enabled) {
      const intent = {
        name: getIntentId(id),
        entities: [],
        contexts: [data.category || 'global'],
        utterances: normalizeQuestions(data.questions)
      };
      await this.checkForDuplicatedQuestions(intent.utterances, id);
      await _axios.default.post(`/mod/nlu/intents`, intent, axiosConfig);
    } else {
      await _axios.default.delete(`/mod/nlu/intents/${getIntentId(id)}`, axiosConfig);
    }

    await this.bp.ghost.forBot(this.botId).upsertFile(this.config.qnaDir, `${id}.json`, JSON.stringify({
      id,
      data
    }, undefined, 2));
    return id;
  }

  async insert(qna, statusCb) {
    const ids = await Promise.mapSeries(_lodash.default.isArray(qna) ? qna : [qna], async (data, i) => {
      const id = getQuestionId(data);

      if (data.enabled) {
        const intent = {
          name: getIntentId(id),
          entities: [],
          contexts: [data.category || 'global'],
          utterances: normalizeQuestions(data.questions)
        };
        await this.checkForDuplicatedQuestions(intent.utterances);
        await _axios.default.post(`/mod/nlu/intents`, intent, (await this.getAxiosConfig()));
      }

      await this.bp.ghost.forBot(this.botId).upsertFile(this.config.qnaDir, `${id}.json`, JSON.stringify({
        id,
        data
      }, undefined, 2));
      statusCb && statusCb(i + 1);
      return id;
    });
    return ids;
  }

  async checkForDuplicatedQuestions(newQuestions, editingQnaId) {
    let allQuestions = await this.fetchAllQuestions();

    if (editingQnaId) {
      // when updating, we remove the question from the check
      allQuestions = allQuestions.filter(q => q.id !== editingQnaId);
    }

    const questionsList = _lodash.default.flatMap(allQuestions, entry => entry.data.questions);

    const dupes = _lodash.default.uniq(_lodash.default.filter(questionsList, question => newQuestions.includes(question)));

    if (dupes.length) {
      throw new Error(`These questions already exists in another entry: ${dupes.join(', ')}`);
    }
  }
  /**
   * This will migrate questions to the new format.
   * @deprecated Questions support multiple answers since v11.3
   */


  migrate_11_2_to_11_3(question) {
    if (!question.data.answers) {
      question.data.answers = [question.data.answer];
    }

    return question;
  }

  async getQuestion(opts) {
    let filename;

    if (typeof opts === 'string') {
      filename = `${opts}.json`;
    } else {
      // opts object
      filename = opts.filename;
    }

    const data = await this.bp.ghost.forBot(this.botId).readFileAsString(this.config.qnaDir, filename);
    return this.migrate_11_2_to_11_3(JSON.parse(data));
  }

  async fetchAllQuestions(opts) {
    try {
      let questions = await this.bp.ghost.forBot(this.botId).directoryListing(this.config.qnaDir, '*.json');

      if (opts && opts.start && opts.count) {
        questions = questions.slice(opts.start, opts.start + opts.count);
      }

      return Promise.map(questions, question => this.getQuestion({
        filename: question
      }));
    } catch (err) {
      this.bp.logger.warn(`Error while reading questions. ${err}`);
      return [];
    }
  }

  async filterByCategoryAndQuestion({
    question,
    categories
  }) {
    const allQuestions = await this.fetchAllQuestions();
    const filteredQuestions = allQuestions.filter(q => {
      const {
        questions,
        category
      } = q.data;
      const isRightId = questions.join('\n').toLowerCase().indexOf(question.toLowerCase()) !== -1;

      if (!categories.length) {
        return isRightId || q.id.includes(question);
      }

      if (!question) {
        return category && categories.indexOf(category) !== -1;
      }

      return isRightId && category && categories.indexOf(category) !== -1;
    });
    return filteredQuestions.reverse();
  }

  async getQuestions({
    question = '',
    categories = []
  }, {
    limit = 50,
    offset = 0
  }) {
    let items = [];
    let count = 0;

    if (!(question || categories.length)) {
      items = await this.fetchAllQuestions({
        start: offset ? parseInt(offset) : undefined,
        count: limit ? parseInt(limit) : undefined
      });
      count = await this.count();
    } else {
      const tmpQuestions = await this.filterByCategoryAndQuestion({
        question,
        categories
      });
      items = tmpQuestions.slice(offset, offset + limit);
      count = tmpQuestions.length;
    }

    return {
      items,
      count
    };
  }

  async count() {
    const questions = await this.fetchAllQuestions();
    return questions.length;
  }

  async delete(qnaId) {
    const ids = _lodash.default.isArray(qnaId) ? qnaId : [qnaId];

    if (ids.length === 0) {
      return;
    }

    const deletePromise = async id => {
      const data = await this.getQuestion(id);

      if (data.data.enabled) {
        await _axios.default.delete(`/mod/nlu/intents/${getIntentId(id)}`, (await this.getAxiosConfig()));
      }

      return this.bp.ghost.forBot(this.botId).deleteFile(this.config.qnaDir, `${id}.json`);
    };

    await Promise.all(ids.map(deletePromise));
  }

  async answersOn(text) {
    const extract = await _axios.default.post('/mod/nlu/extract', {
      text
    }, (await this.getAxiosConfig()));

    const intents = _lodash.default.chain([extract.data['intent'], ...extract.data['intents']]).uniqBy('name').filter(({
      name
    }) => name.startsWith('__qna__')).orderBy(['confidence'], ['desc']).value();

    return Promise.all(intents.map(async ({
      name,
      confidence
    }) => {
      const {
        data: {
          questions,
          answer
        }
      } = await this.getQuestion(name.replace('__qna__', ''));
      return {
        questions,
        answer,
        confidence,
        id: name,
        metadata: []
      };
    }));
  }

  getCategories() {
    return this.categories;
  }

  hasCategories() {
    return this.categories && this.categories.length > 0;
  }

}

exports.default = Storage;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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