"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FiveFolder = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const MapProxy = {
  get: function (target, prop) {
    if (typeof target[prop] === 'undefined') {
      target[prop] = new Map();
    }

    return target[prop];
  }
};
const ZeroProxy = {
  get: function (target, prop) {
    return target[prop] || 0;
  }
};

class FiveFolder {
  constructor(dataset) {
    this.dataset = dataset;

    _defineProperty(this, "results", {});

    _defineProperty(this, "_record", suiteName => (expected, actual) => {
      const {
        tp,
        fp,
        fn,
        confusions
      } = this.results[suiteName];

      if (expected === actual) {
        tp[expected] = tp[expected] + 1;
      } else {
        fp[actual] = fp[actual] + 1;
        fn[expected] = fn[expected] + 1;
        confusions[expected].set(actual, (confusions[expected].get(actual) || 0) + 1);
      }
    });
  }

  async fold(suiteName, trainFn, evaluateFn) {
    this.results[suiteName] = {
      fp: new Proxy({}, ZeroProxy),
      tp: new Proxy({}, ZeroProxy),
      fn: new Proxy({}, ZeroProxy),
      confusions: new Proxy({}, MapProxy)
    };

    const shuffled = _lodash.default.shuffle(this.dataset);

    const chunks = _lodash.default.chunk(shuffled, Math.ceil(shuffled.length / 5));

    await Promise.mapSeries(chunks, async testSet => {
      const trainSet = _lodash.default.flatten(chunks.filter(c => c !== testSet));

      await trainFn([...trainSet]);
      await evaluateFn([...testSet], this._record(suiteName));
    });
  }

  getResults() {
    const ret = {};

    for (const suite in this.results) {
      const classes = _lodash.default.uniq([...Object.keys(this.results[suite].fp), ...Object.keys(this.results[suite].tp), ...Object.keys(this.results[suite].fn)]);

      const result = {};

      for (const cls of classes) {
        const confusions = serializeMap(this.results[suite].confusions[cls]);
        const precision = this.results[suite].tp[cls] / (this.results[suite].tp[cls] + this.results[suite].fp[cls]);
        const recall = this.results[suite].tp[cls] / (this.results[suite].tp[cls] + this.results[suite].fn[cls]);
        const f1 = 2 * (precision * recall / (precision + recall));
        result[cls] = new Proxy({
          tp: this.results[suite].tp[cls],
          fp: this.results[suite].fp[cls],
          fn: this.results[suite].fn[cls],
          samples: this.results[suite].tp[cls] + this.results[suite].fn[cls],
          confusions: confusions,
          precision: isNaN(precision) ? 0 : precision,
          recall: isNaN(recall) ? 0 : recall,
          f1: isNaN(f1) ? 0 : f1
        }, ZeroProxy);
      }

      const v = _lodash.default.values(result);

      result['all'] = {
        f1: _lodash.default.meanBy(v, 'f1'),
        precision: _lodash.default.meanBy(v, 'precision'),
        recall: _lodash.default.meanBy(v, 'recall'),
        tp: 0,
        fp: 0,
        fn: 0
      };
      ret[suite] = result;
    }

    return ret;
  }

}

exports.FiveFolder = FiveFolder;

function serializeMap(map) {
  return Array.from(map.entries()).reduce((acc, curr) => {
    acc[curr[0]] = curr[1];
    return acc;
  }, {});
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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