"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _path = _interopRequireDefault(require("path"));

var _util = require("../util.js");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const N_KEEP_MODELS = 10;

class Storage {
  constructor(config, botId) {
    this.botId = botId;

    _defineProperty(this, "botGhost", void 0);

    _defineProperty(this, "globalGhost", void 0);

    _defineProperty(this, "intentsDir", './intents');

    _defineProperty(this, "entitiesDir", './entities');

    _defineProperty(this, "modelsDir", './models');

    _defineProperty(this, "config", void 0);

    this.config = config;
    this.botGhost = Storage.ghostProvider(this.botId);
    this.globalGhost = Storage.ghostProvider();
  }

  async saveIntent(intent, content) {
    intent = (0, _util.sanitizeFilenameNoExt)(intent);
    const availableEntities = await this.getAvailableEntities();

    if (content.slots) {
      for (const slot of content.slots) {
        // @deprecated > 11 gracefull migration
        if (!slot.entities && slot.entity) {
          slot.entities = [slot.entity];
        }

        for (const entity of slot.entities) {
          if (!availableEntities.find(e => e.name === entity)) {
            throw Error(`"${entity}" is neither a system entity nor a custom entity`);
          }
        }
      }
    }

    if (!content.contexts) {
      content.contexts = ['global'];
    }

    if (intent.length < 1) {
      throw new Error('Invalid intent name, expected at least one character');
    }

    await this.botGhost.upsertFile(this.intentsDir, `${intent}.json`, JSON.stringify(content, undefined, 2));
  }

  async saveConfusionMatrix(modelHash, results) {
    await this.botGhost.upsertFile(this.modelsDir, `confusion__${modelHash}.json`, JSON.stringify(results, undefined, 2), false);
  }

  async getConfusionMatrix(modelHash) {
    return this.botGhost.readFileAsObject(this.modelsDir, `confusion__${modelHash}.json`);
  }

  async deleteIntent(intent) {
    intent = (0, _util.sanitizeFilenameNoExt)(intent);

    if (intent.length < 1) {
      throw new Error('Invalid intent name, expected at least one character');
    }

    try {
      await this.botGhost.deleteFile(this.intentsDir, `${intent}.json`);
    } catch (e) {
      if (e.code !== 'ENOENT' && !e.message.includes("couldn't find")) {
        throw e;
      }
    }
  }

  async getIntents() {
    const intents = await this.botGhost.directoryListing(this.intentsDir, '*.json');
    return Promise.mapSeries(intents, intent => this.getIntent(intent));
  }

  async getIntent(intent) {
    intent = (0, _util.sanitizeFilenameNoExt)(intent);

    if (intent.length < 1) {
      throw new Error('Invalid intent name, expected at least one character');
    }

    const filename = `${intent}.json`;
    const propertiesContent = await this.botGhost.readFileAsString(this.intentsDir, filename);
    let properties = {};

    try {
      properties = JSON.parse(propertiesContent);
    } catch (err) {
      throw new Error(`Could not parse intent properties (invalid JSON). JSON = "${propertiesContent}" in file "${filename}"`);
    }

    const obj = {
      name: intent,
      filename: filename,
      contexts: ['global'],
      // @deprecated remove in bp > 11
      ...properties
    }; // @deprecated remove in bp > 11

    let hasChange = false;

    if (!properties.utterances) {
      await this._legacyAppendIntentUtterances(obj, intent);
      hasChange = true;
    } // @deprecated > 11 graceful migration


    for (const slot of obj.slots || []) {
      if (!slot.entities && slot.entity) {
        slot.entities = [slot.entity];
        delete slot.entity;
        hasChange = true;
      }
    }

    if (hasChange) {
      await this.saveIntent(intent, obj);
    }

    return obj;
  }
  /** @deprecated remove in 12.0+
   * utterances used to be defined in a separate .txt file
   * this is not the case anymore since 11.2
   * we added this method for backward compatibility
   */


  async _legacyAppendIntentUtterances(intent, intentName) {
    const filename = `${intentName}.utterances.txt`;
    const utterancesContent = await this.botGhost.readFileAsString(this.intentsDir, filename);
    intent.utterances = _lodash.default.split(utterancesContent, /\r|\r\n|\n/i).filter(x => x.length);
    await this.botGhost.deleteFile(this.intentsDir, filename);
  }

  async getAvailableEntities() {
    return [...this.getSystemEntities(), ...(await this.getCustomEntities())];
  }

  getSystemEntities() {
    // TODO move this array as static method in DucklingExtractor
    const sysEntNames = !this.config.ducklingEnabled ? [] : ['amountOfMoney', 'distance', 'duration', 'email', 'numeral', 'ordinal', 'phoneNumber', 'quantity', 'temperature', 'time', 'url', 'volume'];
    sysEntNames.unshift('any');
    return sysEntNames.map(e => ({
      name: e,
      type: 'system'
    }));
  }

  async getCustomEntities() {
    const files = await this.botGhost.directoryListing(this.entitiesDir, '*.json');
    return Promise.mapSeries(files, async fileName => {
      const body = await this.botGhost.readFileAsObject(this.entitiesDir, fileName);
      return { ...body,
        id: (0, _util.sanitizeFilenameNoExt)(fileName)
      };
    });
  }

  async saveEntity(entity) {
    const obj = _lodash.default.omit(entity, ['id']);

    return this.botGhost.upsertFile(this.entitiesDir, `${entity.id}.json`, JSON.stringify(obj, undefined, 2));
  }

  async deleteEntity(entityId) {
    return this.botGhost.deleteFile(this.entitiesDir, `${entityId}.json`);
  }

  async _persistModel(model) {
    // TODO Ghost to support streams?
    const modelName = `${model.meta.context}__${model.meta.created_on}__${model.meta.hash}__${model.meta.type}.bin`;
    return this.botGhost.upsertFile(this.modelsDir, modelName, model.model, false);
  }

  async _cleanupModels() {
    const models = await this._getAvailableModels(false);

    const uniqModelMeta = _lodash.default.chain(models).orderBy('created_on', 'desc').uniqBy('hash').value();

    if (uniqModelMeta.length > N_KEEP_MODELS) {
      const threshModel = uniqModelMeta[N_KEEP_MODELS - 1];
      await Promise.all(models.filter(model => model.created_on < threshModel.created_on && model.hash !== threshModel.hash).map(model => this.botGhost.deleteFile(this.modelsDir, model.fileName)));
    }
  }

  async persistModels(models) {
    await Promise.map(models, model => this._persistModel(model));
    return this._cleanupModels();
  }

  async _getAvailableModels(includeGlobalModels = false) {
    const botModels = await this.botGhost.directoryListing(this.modelsDir, '*.+(bin|vec)');
    const globalModels = includeGlobalModels ? await this.globalGhost.directoryListing(this.modelsDir, '*.+(bin|vec)') : [];
    return [...botModels, ...globalModels].map(x => {
      const fileName = _path.default.basename(x);

      const parts = fileName.replace(/\.(bin|vec)$/i, '').split('__');

      if (parts.length !== 4) {
        // we don't support legacy format (old models)
        // this is non-breaking as it will simply re-train the models
        // DEPRECATED – REMOVED THIS CONDITION IN BP > 11
        return undefined;
      }

      return {
        fileName,
        context: parts[0],
        created_on: parseInt(parts[1]) || 0,
        hash: parts[2],
        type: parts[3],
        scope: globalModels.includes(x) ? 'global' : 'bot'
      };
    }).filter(x => !!x);
  }

  async modelExists(modelHash) {
    const models = await this._getAvailableModels(false);
    return !!_lodash.default.find(models, m => m.hash === modelHash);
  }

  async getModelsFromHash(modelHash) {
    const modelsMeta = await this._getAvailableModels(true);
    return Promise.map(modelsMeta.filter(meta => meta.hash === modelHash || meta.scope === 'global'), async meta => {
      const ghostDriver = meta.scope === 'global' ? this.globalGhost : this.botGhost;
      return {
        meta,
        model: await ghostDriver.readFileAsBuffer(this.modelsDir, meta.fileName)
      };
    });
  }

}

exports.default = Storage;

_defineProperty(Storage, "ghostProvider", void 0);
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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