"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _bluebirdRetry = _interopRequireDefault(require("bluebird-retry"));

var _crypto = _interopRequireDefault(require("crypto"));

var _fs = _interopRequireDefault(require("fs"));

var _lodash = _interopRequireWildcard(require("lodash"));

var _ms = _interopRequireDefault(require("ms"));

var _tmp = require("tmp");

var _duckling_extractor = require("./pipelines/entities/duckling_extractor");

var _pattern_extractor = _interopRequireDefault(require("./pipelines/entities/pattern_extractor"));

var _exact_matcher = _interopRequireDefault(require("./pipelines/intents/exact_matcher"));

var _ft_classifier = _interopRequireDefault(require("./pipelines/intents/ft_classifier"));

var _utils = require("./pipelines/intents/utils");

var _ft_lid = require("./pipelines/language/ft_lid");

var _sanitizer = require("./pipelines/language/sanitizer");

var _crf_extractor = _interopRequireDefault(require("./pipelines/slots/crf_extractor"));

var _preProcessor = require("./pipelines/slots/pre-processor");

var _storage = _interopRequireDefault(require("./storage"));

var _typings = require("./typings");

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } else { var newObj = {}; if (obj != null) { for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = Object.defineProperty && Object.getOwnPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : {}; if (desc.get || desc.set) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } } newObj.default = obj; return newObj; } }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const debug = DEBUG('nlu');
const debugExtract = debug.sub('extract');
const debugIntents = debugExtract.sub('intents');
const debugEntities = debugExtract.sub('entities');

class ScopedEngine {
  constructor(logger, botId, config, toolkit) {
    this.logger = logger;
    this.botId = botId;
    this.config = config;
    this.toolkit = toolkit;

    _defineProperty(this, "storage", void 0);

    _defineProperty(this, "confidenceTreshold", 0.7);

    _defineProperty(this, "_preloaded", false);

    _defineProperty(this, "_lmLoaded", false);

    _defineProperty(this, "_currentModelHash", void 0);

    _defineProperty(this, "_exactIntentMatcher", void 0);

    _defineProperty(this, "intentClassifier", void 0);

    _defineProperty(this, "langDetector", void 0);

    _defineProperty(this, "systemEntityExtractor", void 0);

    _defineProperty(this, "slotExtractor", void 0);

    _defineProperty(this, "entityExtractor", void 0);

    _defineProperty(this, "retryPolicy", {
      interval: 100,
      max_interval: 500,
      timeout: 5000,
      max_tries: 3
    });

    _defineProperty(this, "_isSyncing", void 0);

    _defineProperty(this, "_isSyncingTwice", void 0);

    _defineProperty(this, "_autoTrainInterval", 0);

    _defineProperty(this, "_autoTrainTimer", void 0);

    this.storage = new _storage.default(config, this.botId);
    this.intentClassifier = new _ft_classifier.default(toolkit, this.logger, this.config.fastTextOverrides || {});
    this.langDetector = new _ft_lid.FastTextLanguageId(toolkit, this.logger);
    this.systemEntityExtractor = new _duckling_extractor.DucklingEntityExtractor(this.logger);
    this.slotExtractor = new _crf_extractor.default(toolkit);
    this.entityExtractor = new _pattern_extractor.default(toolkit);
    this._autoTrainInterval = (0, _ms.default)(config.autoTrainInterval || 0);
  }

  async init() {
    this.confidenceTreshold = this.config.confidenceTreshold;

    if (isNaN(this.confidenceTreshold) || this.confidenceTreshold < 0 || this.confidenceTreshold > 1) {
      this.confidenceTreshold = 0.7;
    }

    if (this.config.preloadModels) {
      this.sync();
    }

    if (!isNaN(this._autoTrainInterval) && this._autoTrainInterval >= 5000) {
      if (this._autoTrainTimer) {
        clearInterval(this._autoTrainTimer);
      }

      this._autoTrainTimer = setInterval(async () => {
        if (this._preloaded && (await this.checkSyncNeeded())) {
          // Sync only if the model has been already loaded
          this.sync();
        }
      }, this._autoTrainInterval);
    }
  }

  async getIntents() {
    return this.storage.getIntents();
  }
  /**
   * @return The trained model hash
   */


  async sync(forceRetrain = false) {
    if (this._isSyncing) {
      this._isSyncingTwice = true;
      return;
    }

    try {
      this._isSyncing = true;
      const intents = await this.getIntents();

      const modelHash = this._getModelHash(intents);

      let loaded = false;

      if (!forceRetrain && (await this.storage.modelExists(modelHash))) {
        try {
          await this.loadModels(intents, modelHash);
          loaded = true;
        } catch (e) {
          this.logger.attachError(e).warn('Could not load models from storage');
        }
      }

      if (!loaded) {
        this.logger.debug('Retraining model');
        await this.trainModels(intents, modelHash);
        await this.loadModels(intents, modelHash);
      }

      this._currentModelHash = modelHash;
      this._preloaded = true;
    } catch (e) {
      this.logger.attachError(e).error('Could not sync model');
    } finally {
      this._isSyncing = false;

      if (this._isSyncingTwice) {
        this._isSyncingTwice = false;
        return this.sync(); // This floating promise is voluntary
      }
    }

    return this._currentModelHash;
  }

  async extract(text, includedContexts) {
    if (!this._preloaded) {
      await this.sync();
    }

    return (0, _bluebirdRetry.default)(() => this._extract(text, includedContexts), this.retryPolicy);
  }

  async checkSyncNeeded() {
    const intents = await this.storage.getIntents();

    const modelHash = this._getModelHash(intents);

    return intents.length && this._currentModelHash !== modelHash && !this._isSyncing;
  }

  async _loadLanguageModel() {
    if (this._lmLoaded) {
      return;
    } // N/A: we don't care about the hash, we just want the language models which are always returned whatever the hash


    const models = await this.storage.getModelsFromHash('N/A');

    const intentLangModel = _lodash.default.chain(models).filter(model => model.meta.type === _typings.MODEL_TYPES.INTENT_LM).orderBy(model => model.meta.created_on, 'desc').filter(model => model.meta.context === this.config.languageModel).first().value();

    if (intentLangModel && this.intentClassifier instanceof _ft_classifier.default) {
      const fn = (0, _tmp.tmpNameSync)({
        postfix: '.vec'
      });

      _fs.default.writeFileSync(fn, intentLangModel.model);

      this.intentClassifier.prebuiltWordVecPath = fn;
      this.logger.debug(`Using Language Model "${intentLangModel.meta.fileName}"`);
    } else {
      this.logger.warn(`Language model not found for "${this.config.languageModel}"`);
    }

    this._lmLoaded = true;
  }

  async loadModels(intents, modelHash) {
    this.logger.debug(`Restoring models '${modelHash}' from storage`);
    const models = await this.storage.getModelsFromHash(modelHash);

    const intentModels = _lodash.default.chain(models).filter(model => model.meta.type === _typings.MODEL_TYPES.INTENT).orderBy(model => model.meta.created_on, 'desc').uniqBy(model => model.meta.hash + ' ' + model.meta.type + ' ' + model.meta.context).map(x => ({
      name: x.meta.context,
      model: x.model
    })).value();

    const skipgramModel = models.find(model => model.meta.type === _typings.MODEL_TYPES.SLOT_LANG);
    const crfModel = models.find(model => model.meta.type === _typings.MODEL_TYPES.SLOT_CRF);

    if (!intentModels || !intentModels.length || !skipgramModel || !crfModel) {
      throw new Error(`No such model. Hash = "${modelHash}"`);
    }

    const trainingSet = (0, _lodash.flatMap)(intents, intent => {
      return intent.utterances.map(utterance => (0, _preProcessor.generateTrainingSequence)(utterance, intent.slots, intent.name, intent.contexts));
    });
    this._exactIntentMatcher = new _exact_matcher.default(trainingSet);
    await this.intentClassifier.load(intentModels);
    await this.slotExtractor.load(trainingSet, skipgramModel.model, crfModel.model);
    this.logger.debug(`Done restoring models '${modelHash}' from storage`);
  }

  _makeModel(context, hash, model, type) {
    return {
      meta: {
        context,
        created_on: Date.now(),
        hash,
        type,
        scope: 'bot'
      },
      model
    };
  }

  async _trainIntentClassifier(intentDefs, modelHash) {
    this.logger.debug('Training intent classifier');

    try {
      const intentBuff = await this.intentClassifier.train(intentDefs);
      this.logger.debug('Done training intent classifier');
      return intentBuff.map(x => this._makeModel(x.name, modelHash, x.model, _typings.MODEL_TYPES.INTENT));
    } catch (err) {
      this.logger.attachError(err).error('Error training intents');
      throw Error('Unable to train model');
    }
  }

  async _trainSlotTagger(intentDefs, modelHash) {
    this.logger.debug('Training slot tagger');

    try {
      const trainingSet = (0, _lodash.flatMap)(intentDefs, intent => {
        return intent.utterances.map(utterance => (0, _preProcessor.generateTrainingSequence)(utterance, intent.slots, intent.name));
      });
      const {
        language,
        crf
      } = await this.slotExtractor.train(trainingSet);
      this.logger.debug('Done training slot tagger');

      if (language && crf) {
        return [this._makeModel('global', modelHash, language, _typings.MODEL_TYPES.SLOT_LANG), this._makeModel('global', modelHash, crf, _typings.MODEL_TYPES.SLOT_CRF)];
      } else return [];
    } catch (err) {
      this.logger.attachError(err).error('Error training slot tagger');
      throw Error('Unable to train model');
    }
  }

  async trainModels(intentDefs, modelHash) {
    try {
      await this._loadLanguageModel();
      const intentModels = await this._trainIntentClassifier(intentDefs, modelHash);
      const slotTaggerModels = await this._trainSlotTagger(intentDefs, modelHash);
      await this.storage.persistModels([...slotTaggerModels, ...intentModels]);
    } catch (err) {
      this.logger.attachError(err);
    }
  }

  _getModelHash(intents) {
    return _crypto.default.createHash('md5').update(JSON.stringify(intents)).digest('hex');
  }

  async _extractEntities(text, lang) {
    const customEntityDefs = await this.storage.getCustomEntities();
    const patternEntities = await this.entityExtractor.extractPatterns(text, customEntityDefs.filter(ent => ent.type === 'pattern'));
    const listEntities = await this.entityExtractor.extractLists(text, lang, customEntityDefs.filter(ent => ent.type === 'list'));
    const systemEntities = await this.systemEntityExtractor.extract(text, lang);
    debugEntities(text, {
      systemEntities,
      patternEntities,
      listEntities
    });
    return [...systemEntities, ...patternEntities, ...listEntities];
  }

  async _extractIntents(text, includedContexts) {
    const exactIntent = this._exactIntentMatcher.exactMatch((0, _sanitizer.sanitize)(text.toLowerCase()), includedContexts);

    if (exactIntent) {
      return {
        includedContexts,
        intent: exactIntent,
        intents: [exactIntent]
      };
    }

    const intents = await this.intentClassifier.predict(text, includedContexts);
    const intent = (0, _utils.findMostConfidentIntentMeanStd)(intents, this.confidenceTreshold);
    intent.matches = (0, _utils.createIntentMatcher)(intent.name); // alter ctx with the given predictions in case where no ctx were provided

    includedContexts = _lodash.default.chain(intents).map(p => p.context).uniq().value();
    debugIntents(text, {
      intents
    });
    return {
      includedContexts,
      intents,
      intent
    };
  }

  async _extractSlots(text, intent, entities) {
    const intentDef = await this.storage.getIntent(intent.name);
    return await this.slotExtractor.extract(text, intentDef, entities);
  }

  async _extract(text, includedContexts) {
    let ret = {
      errored: true
    };
    const t1 = Date.now();

    try {
      ret.language = await this.langDetector.identify(text);
      ret = { ...ret,
        ...(await this._extractIntents(text, includedContexts))
      };
      ret.entities = await this._extractEntities(text, ret.language);
      ret.slots = await this._extractSlots(text, ret.intent, ret.entities);
      debugEntities('slots', {
        text,
        slots: ret.slots
      });
      ret.errored = false;
    } catch (error) {
      this.logger.attachError(error).error(`Could not extract whole NLU data, ${error}`);
    } finally {
      ret.ms = Date.now() - t1;
      return ret;
    }
  }

}

exports.default = ScopedEngine;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImVuZ2luZS50cyJdLCJuYW1lcyI6WyJkZWJ1ZyIsIkRFQlVHIiwiZGVidWdFeHRyYWN0Iiwic3ViIiwiZGVidWdJbnRlbnRzIiwiZGVidWdFbnRpdGllcyIsIlNjb3BlZEVuZ2luZSIsImNvbnN0cnVjdG9yIiwibG9nZ2VyIiwiYm90SWQiLCJjb25maWciLCJ0b29sa2l0IiwiaW50ZXJ2YWwiLCJtYXhfaW50ZXJ2YWwiLCJ0aW1lb3V0IiwibWF4X3RyaWVzIiwic3RvcmFnZSIsIlN0b3JhZ2UiLCJpbnRlbnRDbGFzc2lmaWVyIiwiRmFzdFRleHRDbGFzc2lmaWVyIiwiZmFzdFRleHRPdmVycmlkZXMiLCJsYW5nRGV0ZWN0b3IiLCJGYXN0VGV4dExhbmd1YWdlSWQiLCJzeXN0ZW1FbnRpdHlFeHRyYWN0b3IiLCJEdWNrbGluZ0VudGl0eUV4dHJhY3RvciIsInNsb3RFeHRyYWN0b3IiLCJDUkZFeHRyYWN0b3IiLCJlbnRpdHlFeHRyYWN0b3IiLCJQYXR0ZXJuRXh0cmFjdG9yIiwiX2F1dG9UcmFpbkludGVydmFsIiwiYXV0b1RyYWluSW50ZXJ2YWwiLCJpbml0IiwiY29uZmlkZW5jZVRyZXNob2xkIiwiaXNOYU4iLCJwcmVsb2FkTW9kZWxzIiwic3luYyIsIl9hdXRvVHJhaW5UaW1lciIsImNsZWFySW50ZXJ2YWwiLCJzZXRJbnRlcnZhbCIsIl9wcmVsb2FkZWQiLCJjaGVja1N5bmNOZWVkZWQiLCJnZXRJbnRlbnRzIiwiZm9yY2VSZXRyYWluIiwiX2lzU3luY2luZyIsIl9pc1N5bmNpbmdUd2ljZSIsImludGVudHMiLCJtb2RlbEhhc2giLCJfZ2V0TW9kZWxIYXNoIiwibG9hZGVkIiwibW9kZWxFeGlzdHMiLCJsb2FkTW9kZWxzIiwiZSIsImF0dGFjaEVycm9yIiwid2FybiIsInRyYWluTW9kZWxzIiwiX2N1cnJlbnRNb2RlbEhhc2giLCJlcnJvciIsImV4dHJhY3QiLCJ0ZXh0IiwiaW5jbHVkZWRDb250ZXh0cyIsIl9leHRyYWN0IiwicmV0cnlQb2xpY3kiLCJsZW5ndGgiLCJfbG9hZExhbmd1YWdlTW9kZWwiLCJfbG1Mb2FkZWQiLCJtb2RlbHMiLCJnZXRNb2RlbHNGcm9tSGFzaCIsImludGVudExhbmdNb2RlbCIsIl8iLCJjaGFpbiIsImZpbHRlciIsIm1vZGVsIiwibWV0YSIsInR5cGUiLCJNT0RFTF9UWVBFUyIsIklOVEVOVF9MTSIsIm9yZGVyQnkiLCJjcmVhdGVkX29uIiwiY29udGV4dCIsImxhbmd1YWdlTW9kZWwiLCJmaXJzdCIsInZhbHVlIiwiZm4iLCJwb3N0Zml4IiwiZnMiLCJ3cml0ZUZpbGVTeW5jIiwicHJlYnVpbHRXb3JkVmVjUGF0aCIsImZpbGVOYW1lIiwiaW50ZW50TW9kZWxzIiwiSU5URU5UIiwidW5pcUJ5IiwiaGFzaCIsIm1hcCIsIngiLCJuYW1lIiwic2tpcGdyYW1Nb2RlbCIsImZpbmQiLCJTTE9UX0xBTkciLCJjcmZNb2RlbCIsIlNMT1RfQ1JGIiwiRXJyb3IiLCJ0cmFpbmluZ1NldCIsImludGVudCIsInV0dGVyYW5jZXMiLCJ1dHRlcmFuY2UiLCJzbG90cyIsImNvbnRleHRzIiwiX2V4YWN0SW50ZW50TWF0Y2hlciIsIkV4YWN0TWF0Y2hlciIsImxvYWQiLCJfbWFrZU1vZGVsIiwiRGF0ZSIsIm5vdyIsInNjb3BlIiwiX3RyYWluSW50ZW50Q2xhc3NpZmllciIsImludGVudERlZnMiLCJpbnRlbnRCdWZmIiwidHJhaW4iLCJlcnIiLCJfdHJhaW5TbG90VGFnZ2VyIiwibGFuZ3VhZ2UiLCJjcmYiLCJzbG90VGFnZ2VyTW9kZWxzIiwicGVyc2lzdE1vZGVscyIsImNyeXB0byIsImNyZWF0ZUhhc2giLCJ1cGRhdGUiLCJKU09OIiwic3RyaW5naWZ5IiwiZGlnZXN0IiwiX2V4dHJhY3RFbnRpdGllcyIsImxhbmciLCJjdXN0b21FbnRpdHlEZWZzIiwiZ2V0Q3VzdG9tRW50aXRpZXMiLCJwYXR0ZXJuRW50aXRpZXMiLCJleHRyYWN0UGF0dGVybnMiLCJlbnQiLCJsaXN0RW50aXRpZXMiLCJleHRyYWN0TGlzdHMiLCJzeXN0ZW1FbnRpdGllcyIsIl9leHRyYWN0SW50ZW50cyIsImV4YWN0SW50ZW50IiwiZXhhY3RNYXRjaCIsInRvTG93ZXJDYXNlIiwicHJlZGljdCIsIm1hdGNoZXMiLCJwIiwidW5pcSIsIl9leHRyYWN0U2xvdHMiLCJlbnRpdGllcyIsImludGVudERlZiIsImdldEludGVudCIsInJldCIsImVycm9yZWQiLCJ0MSIsImlkZW50aWZ5IiwibXMiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7QUFBQTs7QUFFQTs7QUFDQTs7QUFDQTs7QUFFQTs7QUFDQTs7QUFJQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7Ozs7Ozs7QUFFQSxNQUFNQSxLQUFLLEdBQUdDLEtBQUssQ0FBQyxLQUFELENBQW5CO0FBQ0EsTUFBTUMsWUFBWSxHQUFHRixLQUFLLENBQUNHLEdBQU4sQ0FBVSxTQUFWLENBQXJCO0FBQ0EsTUFBTUMsWUFBWSxHQUFHRixZQUFZLENBQUNDLEdBQWIsQ0FBaUIsU0FBakIsQ0FBckI7QUFDQSxNQUFNRSxhQUFhLEdBQUdILFlBQVksQ0FBQ0MsR0FBYixDQUFpQixVQUFqQixDQUF0Qjs7QUFFZSxNQUFNRyxZQUFOLENBQXFDO0FBMkJsREMsRUFBQUEsV0FBVyxDQUNDQyxNQURELEVBRUNDLEtBRkQsRUFHVUMsTUFIVixFQUlBQyxPQUpBLEVBS1Q7QUFBQTtBQUFBO0FBQUE7QUFBQTs7QUFBQTs7QUFBQSxnREE5QmtDLEdBOEJsQzs7QUFBQSx3Q0E1QjRCLEtBNEI1Qjs7QUFBQSx1Q0EzQjJCLEtBMkIzQjs7QUFBQTs7QUFBQTs7QUFBQTs7QUFBQTs7QUFBQTs7QUFBQTs7QUFBQTs7QUFBQSx5Q0FqQm9CO0FBQ3BCQyxNQUFBQSxRQUFRLEVBQUUsR0FEVTtBQUVwQkMsTUFBQUEsWUFBWSxFQUFFLEdBRk07QUFHcEJDLE1BQUFBLE9BQU8sRUFBRSxJQUhXO0FBSXBCQyxNQUFBQSxTQUFTLEVBQUU7QUFKUyxLQWlCcEI7O0FBQUE7O0FBQUE7O0FBQUEsZ0RBUm1DLENBUW5DOztBQUFBOztBQUNBLFNBQUtDLE9BQUwsR0FBZSxJQUFJQyxnQkFBSixDQUFZUCxNQUFaLEVBQW9CLEtBQUtELEtBQXpCLENBQWY7QUFDQSxTQUFLUyxnQkFBTCxHQUF3QixJQUFJQyxzQkFBSixDQUF1QlIsT0FBdkIsRUFBZ0MsS0FBS0gsTUFBckMsRUFBNkMsS0FBS0UsTUFBTCxDQUFZVSxpQkFBWixJQUFpQyxFQUE5RSxDQUF4QjtBQUNBLFNBQUtDLFlBQUwsR0FBb0IsSUFBSUMsMEJBQUosQ0FBdUJYLE9BQXZCLEVBQWdDLEtBQUtILE1BQXJDLENBQXBCO0FBQ0EsU0FBS2UscUJBQUwsR0FBNkIsSUFBSUMsMkNBQUosQ0FBNEIsS0FBS2hCLE1BQWpDLENBQTdCO0FBQ0EsU0FBS2lCLGFBQUwsR0FBcUIsSUFBSUMsc0JBQUosQ0FBaUJmLE9BQWpCLENBQXJCO0FBQ0EsU0FBS2dCLGVBQUwsR0FBdUIsSUFBSUMsMEJBQUosQ0FBcUJqQixPQUFyQixDQUF2QjtBQUNBLFNBQUtrQixrQkFBTCxHQUEwQixpQkFBR25CLE1BQU0sQ0FBQ29CLGlCQUFQLElBQTRCLENBQS9CLENBQTFCO0FBQ0Q7O0FBRUQsUUFBTUMsSUFBTixHQUE0QjtBQUMxQixTQUFLQyxrQkFBTCxHQUEwQixLQUFLdEIsTUFBTCxDQUFZc0Isa0JBQXRDOztBQUVBLFFBQUlDLEtBQUssQ0FBQyxLQUFLRCxrQkFBTixDQUFMLElBQWtDLEtBQUtBLGtCQUFMLEdBQTBCLENBQTVELElBQWlFLEtBQUtBLGtCQUFMLEdBQTBCLENBQS9GLEVBQWtHO0FBQ2hHLFdBQUtBLGtCQUFMLEdBQTBCLEdBQTFCO0FBQ0Q7O0FBRUQsUUFBSSxLQUFLdEIsTUFBTCxDQUFZd0IsYUFBaEIsRUFBK0I7QUFDN0IsV0FBS0MsSUFBTDtBQUNEOztBQUVELFFBQUksQ0FBQ0YsS0FBSyxDQUFDLEtBQUtKLGtCQUFOLENBQU4sSUFBbUMsS0FBS0Esa0JBQUwsSUFBMkIsSUFBbEUsRUFBd0U7QUFDdEUsVUFBSSxLQUFLTyxlQUFULEVBQTBCO0FBQ3hCQyxRQUFBQSxhQUFhLENBQUMsS0FBS0QsZUFBTixDQUFiO0FBQ0Q7O0FBQ0QsV0FBS0EsZUFBTCxHQUF1QkUsV0FBVyxDQUFDLFlBQVk7QUFDN0MsWUFBSSxLQUFLQyxVQUFMLEtBQW9CLE1BQU0sS0FBS0MsZUFBTCxFQUExQixDQUFKLEVBQXVEO0FBQ3JEO0FBQ0EsZUFBS0wsSUFBTDtBQUNEO0FBQ0YsT0FMaUMsRUFLL0IsS0FBS04sa0JBTDBCLENBQWxDO0FBTUQ7QUFDRjs7QUFFRCxRQUFnQlksVUFBaEIsR0FBa0U7QUFDaEUsV0FBTyxLQUFLekIsT0FBTCxDQUFheUIsVUFBYixFQUFQO0FBQ0Q7QUFFRDs7Ozs7QUFHQSxRQUFNTixJQUFOLENBQVdPLFlBQXFCLEdBQUcsS0FBbkMsRUFBMkQ7QUFDekQsUUFBSSxLQUFLQyxVQUFULEVBQXFCO0FBQ25CLFdBQUtDLGVBQUwsR0FBdUIsSUFBdkI7QUFDQTtBQUNEOztBQUVELFFBQUk7QUFDRixXQUFLRCxVQUFMLEdBQWtCLElBQWxCO0FBQ0EsWUFBTUUsT0FBTyxHQUFHLE1BQU0sS0FBS0osVUFBTCxFQUF0Qjs7QUFDQSxZQUFNSyxTQUFTLEdBQUcsS0FBS0MsYUFBTCxDQUFtQkYsT0FBbkIsQ0FBbEI7O0FBQ0EsVUFBSUcsTUFBTSxHQUFHLEtBQWI7O0FBRUEsVUFBSSxDQUFDTixZQUFELEtBQWtCLE1BQU0sS0FBSzFCLE9BQUwsQ0FBYWlDLFdBQWIsQ0FBeUJILFNBQXpCLENBQXhCLENBQUosRUFBa0U7QUFDaEUsWUFBSTtBQUNGLGdCQUFNLEtBQUtJLFVBQUwsQ0FBZ0JMLE9BQWhCLEVBQXlCQyxTQUF6QixDQUFOO0FBQ0FFLFVBQUFBLE1BQU0sR0FBRyxJQUFUO0FBQ0QsU0FIRCxDQUdFLE9BQU9HLENBQVAsRUFBVTtBQUNWLGVBQUszQyxNQUFMLENBQVk0QyxXQUFaLENBQXdCRCxDQUF4QixFQUEyQkUsSUFBM0IsQ0FBZ0Msb0NBQWhDO0FBQ0Q7QUFDRjs7QUFFRCxVQUFJLENBQUNMLE1BQUwsRUFBYTtBQUNYLGFBQUt4QyxNQUFMLENBQVlSLEtBQVosQ0FBa0Isa0JBQWxCO0FBQ0EsY0FBTSxLQUFLc0QsV0FBTCxDQUFpQlQsT0FBakIsRUFBMEJDLFNBQTFCLENBQU47QUFDQSxjQUFNLEtBQUtJLFVBQUwsQ0FBZ0JMLE9BQWhCLEVBQXlCQyxTQUF6QixDQUFOO0FBQ0Q7O0FBRUQsV0FBS1MsaUJBQUwsR0FBeUJULFNBQXpCO0FBQ0EsV0FBS1AsVUFBTCxHQUFrQixJQUFsQjtBQUNELEtBdkJELENBdUJFLE9BQU9ZLENBQVAsRUFBVTtBQUNWLFdBQUszQyxNQUFMLENBQVk0QyxXQUFaLENBQXdCRCxDQUF4QixFQUEyQkssS0FBM0IsQ0FBaUMsc0JBQWpDO0FBQ0QsS0F6QkQsU0F5QlU7QUFDUixXQUFLYixVQUFMLEdBQWtCLEtBQWxCOztBQUNBLFVBQUksS0FBS0MsZUFBVCxFQUEwQjtBQUN4QixhQUFLQSxlQUFMLEdBQXVCLEtBQXZCO0FBQ0EsZUFBTyxLQUFLVCxJQUFMLEVBQVAsQ0FGd0IsQ0FFTDtBQUNwQjtBQUNGOztBQUVELFdBQU8sS0FBS29CLGlCQUFaO0FBQ0Q7O0FBRUQsUUFBTUUsT0FBTixDQUFjQyxJQUFkLEVBQTRCQyxnQkFBNUIsRUFBNEY7QUFDMUYsUUFBSSxDQUFDLEtBQUtwQixVQUFWLEVBQXNCO0FBQ3BCLFlBQU0sS0FBS0osSUFBTCxFQUFOO0FBQ0Q7O0FBRUQsV0FBTyw0QkFBTSxNQUFNLEtBQUt5QixRQUFMLENBQWNGLElBQWQsRUFBb0JDLGdCQUFwQixDQUFaLEVBQW1ELEtBQUtFLFdBQXhELENBQVA7QUFDRDs7QUFFRCxRQUFNckIsZUFBTixHQUEwQztBQUN4QyxVQUFNSyxPQUFPLEdBQUcsTUFBTSxLQUFLN0IsT0FBTCxDQUFheUIsVUFBYixFQUF0Qjs7QUFDQSxVQUFNSyxTQUFTLEdBQUcsS0FBS0MsYUFBTCxDQUFtQkYsT0FBbkIsQ0FBbEI7O0FBRUEsV0FBT0EsT0FBTyxDQUFDaUIsTUFBUixJQUFrQixLQUFLUCxpQkFBTCxLQUEyQlQsU0FBN0MsSUFBMEQsQ0FBQyxLQUFLSCxVQUF2RTtBQUNEOztBQUVELFFBQWNvQixrQkFBZCxHQUFtQztBQUNqQyxRQUFJLEtBQUtDLFNBQVQsRUFBb0I7QUFDbEI7QUFDRCxLQUhnQyxDQUtqQzs7O0FBQ0EsVUFBTUMsTUFBTSxHQUFHLE1BQU0sS0FBS2pELE9BQUwsQ0FBYWtELGlCQUFiLENBQStCLEtBQS9CLENBQXJCOztBQUVBLFVBQU1DLGVBQWUsR0FBR0MsZ0JBQUVDLEtBQUYsQ0FBUUosTUFBUixFQUNyQkssTUFEcUIsQ0FDZEMsS0FBSyxJQUFJQSxLQUFLLENBQUNDLElBQU4sQ0FBV0MsSUFBWCxLQUFvQkMscUJBQVlDLFNBRDNCLEVBRXJCQyxPQUZxQixDQUViTCxLQUFLLElBQUlBLEtBQUssQ0FBQ0MsSUFBTixDQUFXSyxVQUZQLEVBRW1CLE1BRm5CLEVBR3JCUCxNQUhxQixDQUdkQyxLQUFLLElBQUlBLEtBQUssQ0FBQ0MsSUFBTixDQUFXTSxPQUFYLEtBQXVCLEtBQUtwRSxNQUFMLENBQVlxRSxhQUg5QixFQUlyQkMsS0FKcUIsR0FLckJDLEtBTHFCLEVBQXhCOztBQU9BLFFBQUlkLGVBQWUsSUFBSSxLQUFLakQsZ0JBQUwsWUFBaUNDLHNCQUF4RCxFQUE0RTtBQUMxRSxZQUFNK0QsRUFBRSxHQUFHLHNCQUFZO0FBQUVDLFFBQUFBLE9BQU8sRUFBRTtBQUFYLE9BQVosQ0FBWDs7QUFDQUMsa0JBQUdDLGFBQUgsQ0FBaUJILEVBQWpCLEVBQXFCZixlQUFlLENBQUNJLEtBQXJDOztBQUNBLFdBQUtyRCxnQkFBTCxDQUFzQm9FLG1CQUF0QixHQUE0Q0osRUFBNUM7QUFDQSxXQUFLMUUsTUFBTCxDQUFZUixLQUFaLENBQW1CLHlCQUF3Qm1FLGVBQWUsQ0FBQ0ssSUFBaEIsQ0FBcUJlLFFBQVMsR0FBekU7QUFDRCxLQUxELE1BS087QUFDTCxXQUFLL0UsTUFBTCxDQUFZNkMsSUFBWixDQUFrQixpQ0FBZ0MsS0FBSzNDLE1BQUwsQ0FBWXFFLGFBQWMsR0FBNUU7QUFDRDs7QUFFRCxTQUFLZixTQUFMLEdBQWlCLElBQWpCO0FBQ0Q7O0FBRUQsUUFBZ0JkLFVBQWhCLENBQTJCTCxPQUEzQixFQUFnRUMsU0FBaEUsRUFBbUY7QUFDakYsU0FBS3RDLE1BQUwsQ0FBWVIsS0FBWixDQUFtQixxQkFBb0I4QyxTQUFVLGdCQUFqRDtBQUVBLFVBQU1tQixNQUFNLEdBQUcsTUFBTSxLQUFLakQsT0FBTCxDQUFha0QsaUJBQWIsQ0FBK0JwQixTQUEvQixDQUFyQjs7QUFFQSxVQUFNMEMsWUFBWSxHQUFHcEIsZ0JBQUVDLEtBQUYsQ0FBUUosTUFBUixFQUNsQkssTUFEa0IsQ0FDWEMsS0FBSyxJQUFJQSxLQUFLLENBQUNDLElBQU4sQ0FBV0MsSUFBWCxLQUFvQkMscUJBQVllLE1BRDlCLEVBRWxCYixPQUZrQixDQUVWTCxLQUFLLElBQUlBLEtBQUssQ0FBQ0MsSUFBTixDQUFXSyxVQUZWLEVBRXNCLE1BRnRCLEVBR2xCYSxNQUhrQixDQUdYbkIsS0FBSyxJQUFJQSxLQUFLLENBQUNDLElBQU4sQ0FBV21CLElBQVgsR0FBa0IsR0FBbEIsR0FBd0JwQixLQUFLLENBQUNDLElBQU4sQ0FBV0MsSUFBbkMsR0FBMEMsR0FBMUMsR0FBZ0RGLEtBQUssQ0FBQ0MsSUFBTixDQUFXTSxPQUh6RCxFQUlsQmMsR0FKa0IsQ0FJZEMsQ0FBQyxLQUFLO0FBQUVDLE1BQUFBLElBQUksRUFBRUQsQ0FBQyxDQUFDckIsSUFBRixDQUFPTSxPQUFmO0FBQXdCUCxNQUFBQSxLQUFLLEVBQUVzQixDQUFDLENBQUN0QjtBQUFqQyxLQUFMLENBSmEsRUFLbEJVLEtBTGtCLEVBQXJCOztBQU9BLFVBQU1jLGFBQWEsR0FBRzlCLE1BQU0sQ0FBQytCLElBQVAsQ0FBWXpCLEtBQUssSUFBSUEsS0FBSyxDQUFDQyxJQUFOLENBQVdDLElBQVgsS0FBb0JDLHFCQUFZdUIsU0FBckQsQ0FBdEI7QUFDQSxVQUFNQyxRQUFRLEdBQUdqQyxNQUFNLENBQUMrQixJQUFQLENBQVl6QixLQUFLLElBQUlBLEtBQUssQ0FBQ0MsSUFBTixDQUFXQyxJQUFYLEtBQW9CQyxxQkFBWXlCLFFBQXJELENBQWpCOztBQUVBLFFBQUksQ0FBQ1gsWUFBRCxJQUFpQixDQUFDQSxZQUFZLENBQUMxQixNQUEvQixJQUF5QyxDQUFDaUMsYUFBMUMsSUFBMkQsQ0FBQ0csUUFBaEUsRUFBMEU7QUFDeEUsWUFBTSxJQUFJRSxLQUFKLENBQVcsMEJBQXlCdEQsU0FBVSxHQUE5QyxDQUFOO0FBQ0Q7O0FBRUQsVUFBTXVELFdBQVcsR0FBRyxxQkFBUXhELE9BQVIsRUFBaUJ5RCxNQUFNLElBQUk7QUFDN0MsYUFBT0EsTUFBTSxDQUFDQyxVQUFQLENBQWtCWCxHQUFsQixDQUFzQlksU0FBUyxJQUNwQyw0Q0FBeUJBLFNBQXpCLEVBQW9DRixNQUFNLENBQUNHLEtBQTNDLEVBQWtESCxNQUFNLENBQUNSLElBQXpELEVBQStEUSxNQUFNLENBQUNJLFFBQXRFLENBREssQ0FBUDtBQUdELEtBSm1CLENBQXBCO0FBTUEsU0FBS0MsbUJBQUwsR0FBMkIsSUFBSUMsc0JBQUosQ0FBaUJQLFdBQWpCLENBQTNCO0FBQ0EsVUFBTSxLQUFLbkYsZ0JBQUwsQ0FBc0IyRixJQUF0QixDQUEyQnJCLFlBQTNCLENBQU47QUFFQSxVQUFNLEtBQUsvRCxhQUFMLENBQW1Cb0YsSUFBbkIsQ0FBd0JSLFdBQXhCLEVBQXFDTixhQUFhLENBQUN4QixLQUFuRCxFQUEwRDJCLFFBQVEsQ0FBQzNCLEtBQW5FLENBQU47QUFFQSxTQUFLL0QsTUFBTCxDQUFZUixLQUFaLENBQW1CLDBCQUF5QjhDLFNBQVUsZ0JBQXREO0FBQ0Q7O0FBRU9nRSxFQUFBQSxVQUFSLENBQW1CaEMsT0FBbkIsRUFBb0NhLElBQXBDLEVBQWtEcEIsS0FBbEQsRUFBaUVFLElBQWpFLEVBQXNGO0FBQ3BGLFdBQU87QUFDTEQsTUFBQUEsSUFBSSxFQUFFO0FBQ0pNLFFBQUFBLE9BREk7QUFFSkQsUUFBQUEsVUFBVSxFQUFFa0MsSUFBSSxDQUFDQyxHQUFMLEVBRlI7QUFHSnJCLFFBQUFBLElBSEk7QUFJSmxCLFFBQUFBLElBSkk7QUFLSndDLFFBQUFBLEtBQUssRUFBRTtBQUxILE9BREQ7QUFRTDFDLE1BQUFBO0FBUkssS0FBUDtBQVVEOztBQUVELFFBQWMyQyxzQkFBZCxDQUFxQ0MsVUFBckMsRUFBNkVyRSxTQUE3RSxFQUFrSDtBQUNoSCxTQUFLdEMsTUFBTCxDQUFZUixLQUFaLENBQWtCLDRCQUFsQjs7QUFFQSxRQUFJO0FBQ0YsWUFBTW9ILFVBQVUsR0FBRyxNQUFNLEtBQUtsRyxnQkFBTCxDQUFzQm1HLEtBQXRCLENBQTRCRixVQUE1QixDQUF6QjtBQUNBLFdBQUszRyxNQUFMLENBQVlSLEtBQVosQ0FBa0IsaUNBQWxCO0FBQ0EsYUFBT29ILFVBQVUsQ0FBQ3hCLEdBQVgsQ0FBZUMsQ0FBQyxJQUFJLEtBQUtpQixVQUFMLENBQWdCakIsQ0FBQyxDQUFDQyxJQUFsQixFQUF3QmhELFNBQXhCLEVBQW1DK0MsQ0FBQyxDQUFDdEIsS0FBckMsRUFBNENHLHFCQUFZZSxNQUF4RCxDQUFwQixDQUFQO0FBQ0QsS0FKRCxDQUlFLE9BQU82QixHQUFQLEVBQVk7QUFDWixXQUFLOUcsTUFBTCxDQUFZNEMsV0FBWixDQUF3QmtFLEdBQXhCLEVBQTZCOUQsS0FBN0IsQ0FBbUMsd0JBQW5DO0FBQ0EsWUFBTTRDLEtBQUssQ0FBQyx1QkFBRCxDQUFYO0FBQ0Q7QUFDRjs7QUFFRCxRQUFjbUIsZ0JBQWQsQ0FBK0JKLFVBQS9CLEVBQXVFckUsU0FBdkUsRUFBNEc7QUFDMUcsU0FBS3RDLE1BQUwsQ0FBWVIsS0FBWixDQUFrQixzQkFBbEI7O0FBRUEsUUFBSTtBQUNGLFlBQU1xRyxXQUFXLEdBQUcscUJBQVFjLFVBQVIsRUFBb0JiLE1BQU0sSUFBSTtBQUNoRCxlQUFPQSxNQUFNLENBQUNDLFVBQVAsQ0FBa0JYLEdBQWxCLENBQXNCWSxTQUFTLElBQUksNENBQXlCQSxTQUF6QixFQUFvQ0YsTUFBTSxDQUFDRyxLQUEzQyxFQUFrREgsTUFBTSxDQUFDUixJQUF6RCxDQUFuQyxDQUFQO0FBQ0QsT0FGbUIsQ0FBcEI7QUFHQSxZQUFNO0FBQUUwQixRQUFBQSxRQUFGO0FBQVlDLFFBQUFBO0FBQVosVUFBb0IsTUFBTSxLQUFLaEcsYUFBTCxDQUFtQjRGLEtBQW5CLENBQXlCaEIsV0FBekIsQ0FBaEM7QUFDQSxXQUFLN0YsTUFBTCxDQUFZUixLQUFaLENBQWtCLDJCQUFsQjs7QUFFQSxVQUFJd0gsUUFBUSxJQUFJQyxHQUFoQixFQUFxQjtBQUNuQixlQUFPLENBQ0wsS0FBS1gsVUFBTCxDQUFnQixRQUFoQixFQUEwQmhFLFNBQTFCLEVBQXFDMEUsUUFBckMsRUFBK0M5QyxxQkFBWXVCLFNBQTNELENBREssRUFFTCxLQUFLYSxVQUFMLENBQWdCLFFBQWhCLEVBQTBCaEUsU0FBMUIsRUFBcUMyRSxHQUFyQyxFQUEwQy9DLHFCQUFZeUIsUUFBdEQsQ0FGSyxDQUFQO0FBSUQsT0FMRCxNQUtPLE9BQU8sRUFBUDtBQUNSLEtBYkQsQ0FhRSxPQUFPbUIsR0FBUCxFQUFZO0FBQ1osV0FBSzlHLE1BQUwsQ0FBWTRDLFdBQVosQ0FBd0JrRSxHQUF4QixFQUE2QjlELEtBQTdCLENBQW1DLDRCQUFuQztBQUNBLFlBQU00QyxLQUFLLENBQUMsdUJBQUQsQ0FBWDtBQUNEO0FBQ0Y7O0FBRUQsUUFBZ0I5QyxXQUFoQixDQUE0QjZELFVBQTVCLEVBQW9FckUsU0FBcEUsRUFBdUY7QUFDckYsUUFBSTtBQUNGLFlBQU0sS0FBS2lCLGtCQUFMLEVBQU47QUFFQSxZQUFNeUIsWUFBWSxHQUFHLE1BQU0sS0FBSzBCLHNCQUFMLENBQTRCQyxVQUE1QixFQUF3Q3JFLFNBQXhDLENBQTNCO0FBQ0EsWUFBTTRFLGdCQUFnQixHQUFHLE1BQU0sS0FBS0gsZ0JBQUwsQ0FBc0JKLFVBQXRCLEVBQWtDckUsU0FBbEMsQ0FBL0I7QUFFQSxZQUFNLEtBQUs5QixPQUFMLENBQWEyRyxhQUFiLENBQTJCLENBQUMsR0FBR0QsZ0JBQUosRUFBc0IsR0FBR2xDLFlBQXpCLENBQTNCLENBQU47QUFDRCxLQVBELENBT0UsT0FBTzhCLEdBQVAsRUFBWTtBQUNaLFdBQUs5RyxNQUFMLENBQVk0QyxXQUFaLENBQXdCa0UsR0FBeEI7QUFDRDtBQUNGOztBQUVPdkUsRUFBQUEsYUFBUixDQUFzQkYsT0FBdEIsRUFBMkQ7QUFDekQsV0FBTytFLGdCQUNKQyxVQURJLENBQ08sS0FEUCxFQUVKQyxNQUZJLENBRUdDLElBQUksQ0FBQ0MsU0FBTCxDQUFlbkYsT0FBZixDQUZILEVBR0pvRixNQUhJLENBR0csS0FISCxDQUFQO0FBSUQ7O0FBRUQsUUFBY0MsZ0JBQWQsQ0FBK0J4RSxJQUEvQixFQUE2Q3lFLElBQTdDLEVBQXNGO0FBQ3BGLFVBQU1DLGdCQUFnQixHQUFHLE1BQU0sS0FBS3BILE9BQUwsQ0FBYXFILGlCQUFiLEVBQS9CO0FBRUEsVUFBTUMsZUFBZSxHQUFHLE1BQU0sS0FBSzNHLGVBQUwsQ0FBcUI0RyxlQUFyQixDQUM1QjdFLElBRDRCLEVBRTVCMEUsZ0JBQWdCLENBQUM5RCxNQUFqQixDQUF3QmtFLEdBQUcsSUFBSUEsR0FBRyxDQUFDL0QsSUFBSixLQUFhLFNBQTVDLENBRjRCLENBQTlCO0FBS0EsVUFBTWdFLFlBQVksR0FBRyxNQUFNLEtBQUs5RyxlQUFMLENBQXFCK0csWUFBckIsQ0FDekJoRixJQUR5QixFQUV6QnlFLElBRnlCLEVBR3pCQyxnQkFBZ0IsQ0FBQzlELE1BQWpCLENBQXdCa0UsR0FBRyxJQUFJQSxHQUFHLENBQUMvRCxJQUFKLEtBQWEsTUFBNUMsQ0FIeUIsQ0FBM0I7QUFNQSxVQUFNa0UsY0FBYyxHQUFHLE1BQU0sS0FBS3BILHFCQUFMLENBQTJCa0MsT0FBM0IsQ0FBbUNDLElBQW5DLEVBQXlDeUUsSUFBekMsQ0FBN0I7QUFDQTlILElBQUFBLGFBQWEsQ0FBQ3FELElBQUQsRUFBTztBQUFFaUYsTUFBQUEsY0FBRjtBQUFrQkwsTUFBQUEsZUFBbEI7QUFBbUNHLE1BQUFBO0FBQW5DLEtBQVAsQ0FBYjtBQUNBLFdBQU8sQ0FBQyxHQUFHRSxjQUFKLEVBQW9CLEdBQUdMLGVBQXZCLEVBQXdDLEdBQUdHLFlBQTNDLENBQVA7QUFDRDs7QUFFRCxRQUFjRyxlQUFkLENBQ0VsRixJQURGLEVBRUVDLGdCQUZGLEVBRzhGO0FBQzVGLFVBQU1rRixXQUFXLEdBQUcsS0FBS2xDLG1CQUFMLENBQXlCbUMsVUFBekIsQ0FBb0MseUJBQVNwRixJQUFJLENBQUNxRixXQUFMLEVBQVQsQ0FBcEMsRUFBa0VwRixnQkFBbEUsQ0FBcEI7O0FBRUEsUUFBSWtGLFdBQUosRUFBaUI7QUFDZixhQUFPO0FBQ0xsRixRQUFBQSxnQkFESztBQUVMMkMsUUFBQUEsTUFBTSxFQUFFdUMsV0FGSDtBQUdMaEcsUUFBQUEsT0FBTyxFQUFFLENBQUNnRyxXQUFEO0FBSEosT0FBUDtBQUtEOztBQUVELFVBQU1oRyxPQUFPLEdBQUcsTUFBTSxLQUFLM0IsZ0JBQUwsQ0FBc0I4SCxPQUF0QixDQUE4QnRGLElBQTlCLEVBQW9DQyxnQkFBcEMsQ0FBdEI7QUFDQSxVQUFNMkMsTUFBTSxHQUFHLDJDQUErQnpELE9BQS9CLEVBQXdDLEtBQUtiLGtCQUE3QyxDQUFmO0FBQ0FzRSxJQUFBQSxNQUFNLENBQUMyQyxPQUFQLEdBQWlCLGdDQUFvQjNDLE1BQU0sQ0FBQ1IsSUFBM0IsQ0FBakIsQ0FiNEYsQ0FlNUY7O0FBQ0FuQyxJQUFBQSxnQkFBZ0IsR0FBR1MsZ0JBQUVDLEtBQUYsQ0FBUXhCLE9BQVIsRUFDaEIrQyxHQURnQixDQUNac0QsQ0FBQyxJQUFJQSxDQUFDLENBQUNwRSxPQURLLEVBRWhCcUUsSUFGZ0IsR0FHaEJsRSxLQUhnQixFQUFuQjtBQUtBN0UsSUFBQUEsWUFBWSxDQUFDc0QsSUFBRCxFQUFPO0FBQUViLE1BQUFBO0FBQUYsS0FBUCxDQUFaO0FBRUEsV0FBTztBQUNMYyxNQUFBQSxnQkFESztBQUVMZCxNQUFBQSxPQUZLO0FBR0x5RCxNQUFBQTtBQUhLLEtBQVA7QUFLRDs7QUFFRCxRQUFjOEMsYUFBZCxDQUNFMUYsSUFERixFQUVFNEMsTUFGRixFQUdFK0MsUUFIRixFQUlvQztBQUNsQyxVQUFNQyxTQUFTLEdBQUcsTUFBTSxLQUFLdEksT0FBTCxDQUFhdUksU0FBYixDQUF1QmpELE1BQU0sQ0FBQ1IsSUFBOUIsQ0FBeEI7QUFDQSxXQUFPLE1BQU0sS0FBS3JFLGFBQUwsQ0FBbUJnQyxPQUFuQixDQUEyQkMsSUFBM0IsRUFBaUM0RixTQUFqQyxFQUE0Q0QsUUFBNUMsQ0FBYjtBQUNEOztBQUVELFFBQWN6RixRQUFkLENBQXVCRixJQUF2QixFQUFxQ0MsZ0JBQXJDLEVBQXFHO0FBQ25HLFFBQUk2RixHQUFRLEdBQUc7QUFBRUMsTUFBQUEsT0FBTyxFQUFFO0FBQVgsS0FBZjtBQUNBLFVBQU1DLEVBQUUsR0FBRzNDLElBQUksQ0FBQ0MsR0FBTCxFQUFYOztBQUNBLFFBQUk7QUFDRndDLE1BQUFBLEdBQUcsQ0FBQ2hDLFFBQUosR0FBZSxNQUFNLEtBQUtuRyxZQUFMLENBQWtCc0ksUUFBbEIsQ0FBMkJqRyxJQUEzQixDQUFyQjtBQUNBOEYsTUFBQUEsR0FBRyxHQUFHLEVBQUUsR0FBR0EsR0FBTDtBQUFVLFlBQUksTUFBTSxLQUFLWixlQUFMLENBQXFCbEYsSUFBckIsRUFBMkJDLGdCQUEzQixDQUFWO0FBQVYsT0FBTjtBQUNBNkYsTUFBQUEsR0FBRyxDQUFDSCxRQUFKLEdBQWUsTUFBTSxLQUFLbkIsZ0JBQUwsQ0FBc0J4RSxJQUF0QixFQUE0QjhGLEdBQUcsQ0FBQ2hDLFFBQWhDLENBQXJCO0FBQ0FnQyxNQUFBQSxHQUFHLENBQUMvQyxLQUFKLEdBQVksTUFBTSxLQUFLMkMsYUFBTCxDQUFtQjFGLElBQW5CLEVBQXlCOEYsR0FBRyxDQUFDbEQsTUFBN0IsRUFBcUNrRCxHQUFHLENBQUNILFFBQXpDLENBQWxCO0FBQ0FoSixNQUFBQSxhQUFhLENBQUMsT0FBRCxFQUFVO0FBQUVxRCxRQUFBQSxJQUFGO0FBQVErQyxRQUFBQSxLQUFLLEVBQUUrQyxHQUFHLENBQUMvQztBQUFuQixPQUFWLENBQWI7QUFDQStDLE1BQUFBLEdBQUcsQ0FBQ0MsT0FBSixHQUFjLEtBQWQ7QUFDRCxLQVBELENBT0UsT0FBT2pHLEtBQVAsRUFBYztBQUNkLFdBQUtoRCxNQUFMLENBQVk0QyxXQUFaLENBQXdCSSxLQUF4QixFQUErQkEsS0FBL0IsQ0FBc0MscUNBQW9DQSxLQUFNLEVBQWhGO0FBQ0QsS0FURCxTQVNVO0FBQ1JnRyxNQUFBQSxHQUFHLENBQUNJLEVBQUosR0FBUzdDLElBQUksQ0FBQ0MsR0FBTCxLQUFhMEMsRUFBdEI7QUFDQSxhQUFPRixHQUFQO0FBQ0Q7QUFDRjs7QUEvVWlEIiwic291cmNlUm9vdCI6Ii92YXIvbGliL2plbmtpbnMvd29ya3NwYWNlL2J1aWxkLWxpbnV4L21vZHVsZXMvbmx1L3NyYy9iYWNrZW5kIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHJldHJ5IGZyb20gJ2JsdWViaXJkLXJldHJ5J1xuaW1wb3J0ICogYXMgc2RrIGZyb20gJ2JvdHByZXNzL3NkaydcbmltcG9ydCBjcnlwdG8gZnJvbSAnY3J5cHRvJ1xuaW1wb3J0IGZzIGZyb20gJ2ZzJ1xuaW1wb3J0IHsgZmxhdE1hcCB9IGZyb20gJ2xvZGFzaCdcbmltcG9ydCBfIGZyb20gJ2xvZGFzaCdcbmltcG9ydCBtcyBmcm9tICdtcydcbmltcG9ydCB7IHRtcE5hbWVTeW5jIH0gZnJvbSAndG1wJ1xuXG5pbXBvcnQgeyBDb25maWcgfSBmcm9tICcuLi9jb25maWcnXG5cbmltcG9ydCB7IER1Y2tsaW5nRW50aXR5RXh0cmFjdG9yIH0gZnJvbSAnLi9waXBlbGluZXMvZW50aXRpZXMvZHVja2xpbmdfZXh0cmFjdG9yJ1xuaW1wb3J0IFBhdHRlcm5FeHRyYWN0b3IgZnJvbSAnLi9waXBlbGluZXMvZW50aXRpZXMvcGF0dGVybl9leHRyYWN0b3InXG5pbXBvcnQgRXhhY3RNYXRjaGVyIGZyb20gJy4vcGlwZWxpbmVzL2ludGVudHMvZXhhY3RfbWF0Y2hlcidcbmltcG9ydCBGYXN0VGV4dENsYXNzaWZpZXIgZnJvbSAnLi9waXBlbGluZXMvaW50ZW50cy9mdF9jbGFzc2lmaWVyJ1xuaW1wb3J0IHsgY3JlYXRlSW50ZW50TWF0Y2hlciwgZmluZE1vc3RDb25maWRlbnRJbnRlbnRNZWFuU3RkIH0gZnJvbSAnLi9waXBlbGluZXMvaW50ZW50cy91dGlscydcbmltcG9ydCB7IEZhc3RUZXh0TGFuZ3VhZ2VJZCB9IGZyb20gJy4vcGlwZWxpbmVzL2xhbmd1YWdlL2Z0X2xpZCdcbmltcG9ydCB7IHNhbml0aXplIH0gZnJvbSAnLi9waXBlbGluZXMvbGFuZ3VhZ2Uvc2FuaXRpemVyJ1xuaW1wb3J0IENSRkV4dHJhY3RvciBmcm9tICcuL3BpcGVsaW5lcy9zbG90cy9jcmZfZXh0cmFjdG9yJ1xuaW1wb3J0IHsgZ2VuZXJhdGVUcmFpbmluZ1NlcXVlbmNlIH0gZnJvbSAnLi9waXBlbGluZXMvc2xvdHMvcHJlLXByb2Nlc3NvcidcbmltcG9ydCBTdG9yYWdlIGZyb20gJy4vc3RvcmFnZSdcbmltcG9ydCB7IEVuZ2luZSwgRW50aXR5RXh0cmFjdG9yLCBMYW5ndWFnZUlkZW50aWZpZXIsIE1vZGVsLCBNT0RFTF9UWVBFUywgU2xvdEV4dHJhY3RvciB9IGZyb20gJy4vdHlwaW5ncydcblxuY29uc3QgZGVidWcgPSBERUJVRygnbmx1JylcbmNvbnN0IGRlYnVnRXh0cmFjdCA9IGRlYnVnLnN1YignZXh0cmFjdCcpXG5jb25zdCBkZWJ1Z0ludGVudHMgPSBkZWJ1Z0V4dHJhY3Quc3ViKCdpbnRlbnRzJylcbmNvbnN0IGRlYnVnRW50aXRpZXMgPSBkZWJ1Z0V4dHJhY3Quc3ViKCdlbnRpdGllcycpXG5cbmV4cG9ydCBkZWZhdWx0IGNsYXNzIFNjb3BlZEVuZ2luZSBpbXBsZW1lbnRzIEVuZ2luZSB7XG4gIHB1YmxpYyByZWFkb25seSBzdG9yYWdlOiBTdG9yYWdlXG4gIHB1YmxpYyBjb25maWRlbmNlVHJlc2hvbGQ6IG51bWJlciA9IDAuN1xuXG4gIHByaXZhdGUgX3ByZWxvYWRlZDogYm9vbGVhbiA9IGZhbHNlXG4gIHByaXZhdGUgX2xtTG9hZGVkOiBib29sZWFuID0gZmFsc2VcbiAgcHJpdmF0ZSBfY3VycmVudE1vZGVsSGFzaDogc3RyaW5nXG4gIHByaXZhdGUgX2V4YWN0SW50ZW50TWF0Y2hlcjogRXhhY3RNYXRjaGVyXG5cbiAgcHJpdmF0ZSByZWFkb25seSBpbnRlbnRDbGFzc2lmaWVyOiBGYXN0VGV4dENsYXNzaWZpZXJcbiAgcHJpdmF0ZSByZWFkb25seSBsYW5nRGV0ZWN0b3I6IExhbmd1YWdlSWRlbnRpZmllclxuICBwcml2YXRlIHJlYWRvbmx5IHN5c3RlbUVudGl0eUV4dHJhY3RvcjogRW50aXR5RXh0cmFjdG9yXG4gIHByaXZhdGUgcmVhZG9ubHkgc2xvdEV4dHJhY3RvcjogU2xvdEV4dHJhY3RvclxuICBwcml2YXRlIHJlYWRvbmx5IGVudGl0eUV4dHJhY3RvcjogUGF0dGVybkV4dHJhY3RvclxuXG4gIHByaXZhdGUgcmV0cnlQb2xpY3kgPSB7XG4gICAgaW50ZXJ2YWw6IDEwMCxcbiAgICBtYXhfaW50ZXJ2YWw6IDUwMCxcbiAgICB0aW1lb3V0OiA1MDAwLFxuICAgIG1heF90cmllczogM1xuICB9XG5cbiAgcHJpdmF0ZSBfaXNTeW5jaW5nOiBib29sZWFuXG4gIHByaXZhdGUgX2lzU3luY2luZ1R3aWNlOiBib29sZWFuXG4gIHByaXZhdGUgX2F1dG9UcmFpbkludGVydmFsOiBudW1iZXIgPSAwXG4gIHByaXZhdGUgX2F1dG9UcmFpblRpbWVyOiBOb2RlSlMuVGltZXJcblxuICBjb25zdHJ1Y3RvcihcbiAgICBwcm90ZWN0ZWQgbG9nZ2VyOiBzZGsuTG9nZ2VyLFxuICAgIHByb3RlY3RlZCBib3RJZDogc3RyaW5nLFxuICAgIHByb3RlY3RlZCByZWFkb25seSBjb25maWc6IENvbmZpZyxcbiAgICByZWFkb25seSB0b29sa2l0OiB0eXBlb2Ygc2RrLk1MVG9vbGtpdFxuICApIHtcbiAgICB0aGlzLnN0b3JhZ2UgPSBuZXcgU3RvcmFnZShjb25maWcsIHRoaXMuYm90SWQpXG4gICAgdGhpcy5pbnRlbnRDbGFzc2lmaWVyID0gbmV3IEZhc3RUZXh0Q2xhc3NpZmllcih0b29sa2l0LCB0aGlzLmxvZ2dlciwgdGhpcy5jb25maWcuZmFzdFRleHRPdmVycmlkZXMgfHwge30pXG4gICAgdGhpcy5sYW5nRGV0ZWN0b3IgPSBuZXcgRmFzdFRleHRMYW5ndWFnZUlkKHRvb2xraXQsIHRoaXMubG9nZ2VyKVxuICAgIHRoaXMuc3lzdGVtRW50aXR5RXh0cmFjdG9yID0gbmV3IER1Y2tsaW5nRW50aXR5RXh0cmFjdG9yKHRoaXMubG9nZ2VyKVxuICAgIHRoaXMuc2xvdEV4dHJhY3RvciA9IG5ldyBDUkZFeHRyYWN0b3IodG9vbGtpdClcbiAgICB0aGlzLmVudGl0eUV4dHJhY3RvciA9IG5ldyBQYXR0ZXJuRXh0cmFjdG9yKHRvb2xraXQpXG4gICAgdGhpcy5fYXV0b1RyYWluSW50ZXJ2YWwgPSBtcyhjb25maWcuYXV0b1RyYWluSW50ZXJ2YWwgfHwgMClcbiAgfVxuXG4gIGFzeW5jIGluaXQoKTogUHJvbWlzZTx2b2lkPiB7XG4gICAgdGhpcy5jb25maWRlbmNlVHJlc2hvbGQgPSB0aGlzLmNvbmZpZy5jb25maWRlbmNlVHJlc2hvbGRcblxuICAgIGlmIChpc05hTih0aGlzLmNvbmZpZGVuY2VUcmVzaG9sZCkgfHwgdGhpcy5jb25maWRlbmNlVHJlc2hvbGQgPCAwIHx8IHRoaXMuY29uZmlkZW5jZVRyZXNob2xkID4gMSkge1xuICAgICAgdGhpcy5jb25maWRlbmNlVHJlc2hvbGQgPSAwLjdcbiAgICB9XG5cbiAgICBpZiAodGhpcy5jb25maWcucHJlbG9hZE1vZGVscykge1xuICAgICAgdGhpcy5zeW5jKClcbiAgICB9XG5cbiAgICBpZiAoIWlzTmFOKHRoaXMuX2F1dG9UcmFpbkludGVydmFsKSAmJiB0aGlzLl9hdXRvVHJhaW5JbnRlcnZhbCA+PSA1MDAwKSB7XG4gICAgICBpZiAodGhpcy5fYXV0b1RyYWluVGltZXIpIHtcbiAgICAgICAgY2xlYXJJbnRlcnZhbCh0aGlzLl9hdXRvVHJhaW5UaW1lcilcbiAgICAgIH1cbiAgICAgIHRoaXMuX2F1dG9UcmFpblRpbWVyID0gc2V0SW50ZXJ2YWwoYXN5bmMgKCkgPT4ge1xuICAgICAgICBpZiAodGhpcy5fcHJlbG9hZGVkICYmIChhd2FpdCB0aGlzLmNoZWNrU3luY05lZWRlZCgpKSkge1xuICAgICAgICAgIC8vIFN5bmMgb25seSBpZiB0aGUgbW9kZWwgaGFzIGJlZW4gYWxyZWFkeSBsb2FkZWRcbiAgICAgICAgICB0aGlzLnN5bmMoKVxuICAgICAgICB9XG4gICAgICB9LCB0aGlzLl9hdXRvVHJhaW5JbnRlcnZhbClcbiAgICB9XG4gIH1cblxuICBwcm90ZWN0ZWQgYXN5bmMgZ2V0SW50ZW50cygpOiBQcm9taXNlPHNkay5OTFUuSW50ZW50RGVmaW5pdGlvbltdPiB7XG4gICAgcmV0dXJuIHRoaXMuc3RvcmFnZS5nZXRJbnRlbnRzKClcbiAgfVxuXG4gIC8qKlxuICAgKiBAcmV0dXJuIFRoZSB0cmFpbmVkIG1vZGVsIGhhc2hcbiAgICovXG4gIGFzeW5jIHN5bmMoZm9yY2VSZXRyYWluOiBib29sZWFuID0gZmFsc2UpOiBQcm9taXNlPHN0cmluZz4ge1xuICAgIGlmICh0aGlzLl9pc1N5bmNpbmcpIHtcbiAgICAgIHRoaXMuX2lzU3luY2luZ1R3aWNlID0gdHJ1ZVxuICAgICAgcmV0dXJuXG4gICAgfVxuXG4gICAgdHJ5IHtcbiAgICAgIHRoaXMuX2lzU3luY2luZyA9IHRydWVcbiAgICAgIGNvbnN0IGludGVudHMgPSBhd2FpdCB0aGlzLmdldEludGVudHMoKVxuICAgICAgY29uc3QgbW9kZWxIYXNoID0gdGhpcy5fZ2V0TW9kZWxIYXNoKGludGVudHMpXG4gICAgICBsZXQgbG9hZGVkID0gZmFsc2VcblxuICAgICAgaWYgKCFmb3JjZVJldHJhaW4gJiYgKGF3YWl0IHRoaXMuc3RvcmFnZS5tb2RlbEV4aXN0cyhtb2RlbEhhc2gpKSkge1xuICAgICAgICB0cnkge1xuICAgICAgICAgIGF3YWl0IHRoaXMubG9hZE1vZGVscyhpbnRlbnRzLCBtb2RlbEhhc2gpXG4gICAgICAgICAgbG9hZGVkID0gdHJ1ZVxuICAgICAgICB9IGNhdGNoIChlKSB7XG4gICAgICAgICAgdGhpcy5sb2dnZXIuYXR0YWNoRXJyb3IoZSkud2FybignQ291bGQgbm90IGxvYWQgbW9kZWxzIGZyb20gc3RvcmFnZScpXG4gICAgICAgIH1cbiAgICAgIH1cblxuICAgICAgaWYgKCFsb2FkZWQpIHtcbiAgICAgICAgdGhpcy5sb2dnZXIuZGVidWcoJ1JldHJhaW5pbmcgbW9kZWwnKVxuICAgICAgICBhd2FpdCB0aGlzLnRyYWluTW9kZWxzKGludGVudHMsIG1vZGVsSGFzaClcbiAgICAgICAgYXdhaXQgdGhpcy5sb2FkTW9kZWxzKGludGVudHMsIG1vZGVsSGFzaClcbiAgICAgIH1cblxuICAgICAgdGhpcy5fY3VycmVudE1vZGVsSGFzaCA9IG1vZGVsSGFzaFxuICAgICAgdGhpcy5fcHJlbG9hZGVkID0gdHJ1ZVxuICAgIH0gY2F0Y2ggKGUpIHtcbiAgICAgIHRoaXMubG9nZ2VyLmF0dGFjaEVycm9yKGUpLmVycm9yKCdDb3VsZCBub3Qgc3luYyBtb2RlbCcpXG4gICAgfSBmaW5hbGx5IHtcbiAgICAgIHRoaXMuX2lzU3luY2luZyA9IGZhbHNlXG4gICAgICBpZiAodGhpcy5faXNTeW5jaW5nVHdpY2UpIHtcbiAgICAgICAgdGhpcy5faXNTeW5jaW5nVHdpY2UgPSBmYWxzZVxuICAgICAgICByZXR1cm4gdGhpcy5zeW5jKCkgLy8gVGhpcyBmbG9hdGluZyBwcm9taXNlIGlzIHZvbHVudGFyeVxuICAgICAgfVxuICAgIH1cblxuICAgIHJldHVybiB0aGlzLl9jdXJyZW50TW9kZWxIYXNoXG4gIH1cblxuICBhc3luYyBleHRyYWN0KHRleHQ6IHN0cmluZywgaW5jbHVkZWRDb250ZXh0czogc3RyaW5nW10pOiBQcm9taXNlPHNkay5JTy5FdmVudFVuZGVyc3RhbmRpbmc+IHtcbiAgICBpZiAoIXRoaXMuX3ByZWxvYWRlZCkge1xuICAgICAgYXdhaXQgdGhpcy5zeW5jKClcbiAgICB9XG5cbiAgICByZXR1cm4gcmV0cnkoKCkgPT4gdGhpcy5fZXh0cmFjdCh0ZXh0LCBpbmNsdWRlZENvbnRleHRzKSwgdGhpcy5yZXRyeVBvbGljeSlcbiAgfVxuXG4gIGFzeW5jIGNoZWNrU3luY05lZWRlZCgpOiBQcm9taXNlPGJvb2xlYW4+IHtcbiAgICBjb25zdCBpbnRlbnRzID0gYXdhaXQgdGhpcy5zdG9yYWdlLmdldEludGVudHMoKVxuICAgIGNvbnN0IG1vZGVsSGFzaCA9IHRoaXMuX2dldE1vZGVsSGFzaChpbnRlbnRzKVxuXG4gICAgcmV0dXJuIGludGVudHMubGVuZ3RoICYmIHRoaXMuX2N1cnJlbnRNb2RlbEhhc2ggIT09IG1vZGVsSGFzaCAmJiAhdGhpcy5faXNTeW5jaW5nXG4gIH1cblxuICBwcml2YXRlIGFzeW5jIF9sb2FkTGFuZ3VhZ2VNb2RlbCgpIHtcbiAgICBpZiAodGhpcy5fbG1Mb2FkZWQpIHtcbiAgICAgIHJldHVyblxuICAgIH1cblxuICAgIC8vIE4vQTogd2UgZG9uJ3QgY2FyZSBhYm91dCB0aGUgaGFzaCwgd2UganVzdCB3YW50IHRoZSBsYW5ndWFnZSBtb2RlbHMgd2hpY2ggYXJlIGFsd2F5cyByZXR1cm5lZCB3aGF0ZXZlciB0aGUgaGFzaFxuICAgIGNvbnN0IG1vZGVscyA9IGF3YWl0IHRoaXMuc3RvcmFnZS5nZXRNb2RlbHNGcm9tSGFzaCgnTi9BJylcblxuICAgIGNvbnN0IGludGVudExhbmdNb2RlbCA9IF8uY2hhaW4obW9kZWxzKVxuICAgICAgLmZpbHRlcihtb2RlbCA9PiBtb2RlbC5tZXRhLnR5cGUgPT09IE1PREVMX1RZUEVTLklOVEVOVF9MTSlcbiAgICAgIC5vcmRlckJ5KG1vZGVsID0+IG1vZGVsLm1ldGEuY3JlYXRlZF9vbiwgJ2Rlc2MnKVxuICAgICAgLmZpbHRlcihtb2RlbCA9PiBtb2RlbC5tZXRhLmNvbnRleHQgPT09IHRoaXMuY29uZmlnLmxhbmd1YWdlTW9kZWwpXG4gICAgICAuZmlyc3QoKVxuICAgICAgLnZhbHVlKClcblxuICAgIGlmIChpbnRlbnRMYW5nTW9kZWwgJiYgdGhpcy5pbnRlbnRDbGFzc2lmaWVyIGluc3RhbmNlb2YgRmFzdFRleHRDbGFzc2lmaWVyKSB7XG4gICAgICBjb25zdCBmbiA9IHRtcE5hbWVTeW5jKHsgcG9zdGZpeDogJy52ZWMnIH0pXG4gICAgICBmcy53cml0ZUZpbGVTeW5jKGZuLCBpbnRlbnRMYW5nTW9kZWwubW9kZWwpXG4gICAgICB0aGlzLmludGVudENsYXNzaWZpZXIucHJlYnVpbHRXb3JkVmVjUGF0aCA9IGZuXG4gICAgICB0aGlzLmxvZ2dlci5kZWJ1ZyhgVXNpbmcgTGFuZ3VhZ2UgTW9kZWwgXCIke2ludGVudExhbmdNb2RlbC5tZXRhLmZpbGVOYW1lfVwiYClcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5sb2dnZXIud2FybihgTGFuZ3VhZ2UgbW9kZWwgbm90IGZvdW5kIGZvciBcIiR7dGhpcy5jb25maWcubGFuZ3VhZ2VNb2RlbH1cImApXG4gICAgfVxuXG4gICAgdGhpcy5fbG1Mb2FkZWQgPSB0cnVlXG4gIH1cblxuICBwcm90ZWN0ZWQgYXN5bmMgbG9hZE1vZGVscyhpbnRlbnRzOiBzZGsuTkxVLkludGVudERlZmluaXRpb25bXSwgbW9kZWxIYXNoOiBzdHJpbmcpIHtcbiAgICB0aGlzLmxvZ2dlci5kZWJ1ZyhgUmVzdG9yaW5nIG1vZGVscyAnJHttb2RlbEhhc2h9JyBmcm9tIHN0b3JhZ2VgKVxuXG4gICAgY29uc3QgbW9kZWxzID0gYXdhaXQgdGhpcy5zdG9yYWdlLmdldE1vZGVsc0Zyb21IYXNoKG1vZGVsSGFzaClcblxuICAgIGNvbnN0IGludGVudE1vZGVscyA9IF8uY2hhaW4obW9kZWxzKVxuICAgICAgLmZpbHRlcihtb2RlbCA9PiBtb2RlbC5tZXRhLnR5cGUgPT09IE1PREVMX1RZUEVTLklOVEVOVClcbiAgICAgIC5vcmRlckJ5KG1vZGVsID0+IG1vZGVsLm1ldGEuY3JlYXRlZF9vbiwgJ2Rlc2MnKVxuICAgICAgLnVuaXFCeShtb2RlbCA9PiBtb2RlbC5tZXRhLmhhc2ggKyAnICcgKyBtb2RlbC5tZXRhLnR5cGUgKyAnICcgKyBtb2RlbC5tZXRhLmNvbnRleHQpXG4gICAgICAubWFwKHggPT4gKHsgbmFtZTogeC5tZXRhLmNvbnRleHQsIG1vZGVsOiB4Lm1vZGVsIH0pKVxuICAgICAgLnZhbHVlKClcblxuICAgIGNvbnN0IHNraXBncmFtTW9kZWwgPSBtb2RlbHMuZmluZChtb2RlbCA9PiBtb2RlbC5tZXRhLnR5cGUgPT09IE1PREVMX1RZUEVTLlNMT1RfTEFORylcbiAgICBjb25zdCBjcmZNb2RlbCA9IG1vZGVscy5maW5kKG1vZGVsID0+IG1vZGVsLm1ldGEudHlwZSA9PT0gTU9ERUxfVFlQRVMuU0xPVF9DUkYpXG5cbiAgICBpZiAoIWludGVudE1vZGVscyB8fCAhaW50ZW50TW9kZWxzLmxlbmd0aCB8fCAhc2tpcGdyYW1Nb2RlbCB8fCAhY3JmTW9kZWwpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgTm8gc3VjaCBtb2RlbC4gSGFzaCA9IFwiJHttb2RlbEhhc2h9XCJgKVxuICAgIH1cblxuICAgIGNvbnN0IHRyYWluaW5nU2V0ID0gZmxhdE1hcChpbnRlbnRzLCBpbnRlbnQgPT4ge1xuICAgICAgcmV0dXJuIGludGVudC51dHRlcmFuY2VzLm1hcCh1dHRlcmFuY2UgPT5cbiAgICAgICAgZ2VuZXJhdGVUcmFpbmluZ1NlcXVlbmNlKHV0dGVyYW5jZSwgaW50ZW50LnNsb3RzLCBpbnRlbnQubmFtZSwgaW50ZW50LmNvbnRleHRzKVxuICAgICAgKVxuICAgIH0pXG5cbiAgICB0aGlzLl9leGFjdEludGVudE1hdGNoZXIgPSBuZXcgRXhhY3RNYXRjaGVyKHRyYWluaW5nU2V0KVxuICAgIGF3YWl0IHRoaXMuaW50ZW50Q2xhc3NpZmllci5sb2FkKGludGVudE1vZGVscylcblxuICAgIGF3YWl0IHRoaXMuc2xvdEV4dHJhY3Rvci5sb2FkKHRyYWluaW5nU2V0LCBza2lwZ3JhbU1vZGVsLm1vZGVsLCBjcmZNb2RlbC5tb2RlbClcblxuICAgIHRoaXMubG9nZ2VyLmRlYnVnKGBEb25lIHJlc3RvcmluZyBtb2RlbHMgJyR7bW9kZWxIYXNofScgZnJvbSBzdG9yYWdlYClcbiAgfVxuXG4gIHByaXZhdGUgX21ha2VNb2RlbChjb250ZXh0OiBzdHJpbmcsIGhhc2g6IHN0cmluZywgbW9kZWw6IEJ1ZmZlciwgdHlwZTogc3RyaW5nKTogTW9kZWwge1xuICAgIHJldHVybiB7XG4gICAgICBtZXRhOiB7XG4gICAgICAgIGNvbnRleHQsXG4gICAgICAgIGNyZWF0ZWRfb246IERhdGUubm93KCksXG4gICAgICAgIGhhc2gsXG4gICAgICAgIHR5cGUsXG4gICAgICAgIHNjb3BlOiAnYm90J1xuICAgICAgfSxcbiAgICAgIG1vZGVsXG4gICAgfVxuICB9XG5cbiAgcHJpdmF0ZSBhc3luYyBfdHJhaW5JbnRlbnRDbGFzc2lmaWVyKGludGVudERlZnM6IHNkay5OTFUuSW50ZW50RGVmaW5pdGlvbltdLCBtb2RlbEhhc2g6IHN0cmluZyk6IFByb21pc2U8TW9kZWxbXT4ge1xuICAgIHRoaXMubG9nZ2VyLmRlYnVnKCdUcmFpbmluZyBpbnRlbnQgY2xhc3NpZmllcicpXG5cbiAgICB0cnkge1xuICAgICAgY29uc3QgaW50ZW50QnVmZiA9IGF3YWl0IHRoaXMuaW50ZW50Q2xhc3NpZmllci50cmFpbihpbnRlbnREZWZzKVxuICAgICAgdGhpcy5sb2dnZXIuZGVidWcoJ0RvbmUgdHJhaW5pbmcgaW50ZW50IGNsYXNzaWZpZXInKVxuICAgICAgcmV0dXJuIGludGVudEJ1ZmYubWFwKHggPT4gdGhpcy5fbWFrZU1vZGVsKHgubmFtZSwgbW9kZWxIYXNoLCB4Lm1vZGVsLCBNT0RFTF9UWVBFUy5JTlRFTlQpKVxuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgdGhpcy5sb2dnZXIuYXR0YWNoRXJyb3IoZXJyKS5lcnJvcignRXJyb3IgdHJhaW5pbmcgaW50ZW50cycpXG4gICAgICB0aHJvdyBFcnJvcignVW5hYmxlIHRvIHRyYWluIG1vZGVsJylcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGFzeW5jIF90cmFpblNsb3RUYWdnZXIoaW50ZW50RGVmczogc2RrLk5MVS5JbnRlbnREZWZpbml0aW9uW10sIG1vZGVsSGFzaDogc3RyaW5nKTogUHJvbWlzZTxNb2RlbFtdPiB7XG4gICAgdGhpcy5sb2dnZXIuZGVidWcoJ1RyYWluaW5nIHNsb3QgdGFnZ2VyJylcblxuICAgIHRyeSB7XG4gICAgICBjb25zdCB0cmFpbmluZ1NldCA9IGZsYXRNYXAoaW50ZW50RGVmcywgaW50ZW50ID0+IHtcbiAgICAgICAgcmV0dXJuIGludGVudC51dHRlcmFuY2VzLm1hcCh1dHRlcmFuY2UgPT4gZ2VuZXJhdGVUcmFpbmluZ1NlcXVlbmNlKHV0dGVyYW5jZSwgaW50ZW50LnNsb3RzLCBpbnRlbnQubmFtZSkpXG4gICAgICB9KVxuICAgICAgY29uc3QgeyBsYW5ndWFnZSwgY3JmIH0gPSBhd2FpdCB0aGlzLnNsb3RFeHRyYWN0b3IudHJhaW4odHJhaW5pbmdTZXQpXG4gICAgICB0aGlzLmxvZ2dlci5kZWJ1ZygnRG9uZSB0cmFpbmluZyBzbG90IHRhZ2dlcicpXG5cbiAgICAgIGlmIChsYW5ndWFnZSAmJiBjcmYpIHtcbiAgICAgICAgcmV0dXJuIFtcbiAgICAgICAgICB0aGlzLl9tYWtlTW9kZWwoJ2dsb2JhbCcsIG1vZGVsSGFzaCwgbGFuZ3VhZ2UsIE1PREVMX1RZUEVTLlNMT1RfTEFORyksXG4gICAgICAgICAgdGhpcy5fbWFrZU1vZGVsKCdnbG9iYWwnLCBtb2RlbEhhc2gsIGNyZiwgTU9ERUxfVFlQRVMuU0xPVF9DUkYpXG4gICAgICAgIF1cbiAgICAgIH0gZWxzZSByZXR1cm4gW11cbiAgICB9IGNhdGNoIChlcnIpIHtcbiAgICAgIHRoaXMubG9nZ2VyLmF0dGFjaEVycm9yKGVycikuZXJyb3IoJ0Vycm9yIHRyYWluaW5nIHNsb3QgdGFnZ2VyJylcbiAgICAgIHRocm93IEVycm9yKCdVbmFibGUgdG8gdHJhaW4gbW9kZWwnKVxuICAgIH1cbiAgfVxuXG4gIHByb3RlY3RlZCBhc3luYyB0cmFpbk1vZGVscyhpbnRlbnREZWZzOiBzZGsuTkxVLkludGVudERlZmluaXRpb25bXSwgbW9kZWxIYXNoOiBzdHJpbmcpIHtcbiAgICB0cnkge1xuICAgICAgYXdhaXQgdGhpcy5fbG9hZExhbmd1YWdlTW9kZWwoKVxuXG4gICAgICBjb25zdCBpbnRlbnRNb2RlbHMgPSBhd2FpdCB0aGlzLl90cmFpbkludGVudENsYXNzaWZpZXIoaW50ZW50RGVmcywgbW9kZWxIYXNoKVxuICAgICAgY29uc3Qgc2xvdFRhZ2dlck1vZGVscyA9IGF3YWl0IHRoaXMuX3RyYWluU2xvdFRhZ2dlcihpbnRlbnREZWZzLCBtb2RlbEhhc2gpXG5cbiAgICAgIGF3YWl0IHRoaXMuc3RvcmFnZS5wZXJzaXN0TW9kZWxzKFsuLi5zbG90VGFnZ2VyTW9kZWxzLCAuLi5pbnRlbnRNb2RlbHNdKVxuICAgIH0gY2F0Y2ggKGVycikge1xuICAgICAgdGhpcy5sb2dnZXIuYXR0YWNoRXJyb3IoZXJyKVxuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgX2dldE1vZGVsSGFzaChpbnRlbnRzOiBzZGsuTkxVLkludGVudERlZmluaXRpb25bXSkge1xuICAgIHJldHVybiBjcnlwdG9cbiAgICAgIC5jcmVhdGVIYXNoKCdtZDUnKVxuICAgICAgLnVwZGF0ZShKU09OLnN0cmluZ2lmeShpbnRlbnRzKSlcbiAgICAgIC5kaWdlc3QoJ2hleCcpXG4gIH1cblxuICBwcml2YXRlIGFzeW5jIF9leHRyYWN0RW50aXRpZXModGV4dDogc3RyaW5nLCBsYW5nOiBzdHJpbmcpOiBQcm9taXNlPHNkay5OTFUuRW50aXR5W10+IHtcbiAgICBjb25zdCBjdXN0b21FbnRpdHlEZWZzID0gYXdhaXQgdGhpcy5zdG9yYWdlLmdldEN1c3RvbUVudGl0aWVzKClcblxuICAgIGNvbnN0IHBhdHRlcm5FbnRpdGllcyA9IGF3YWl0IHRoaXMuZW50aXR5RXh0cmFjdG9yLmV4dHJhY3RQYXR0ZXJucyhcbiAgICAgIHRleHQsXG4gICAgICBjdXN0b21FbnRpdHlEZWZzLmZpbHRlcihlbnQgPT4gZW50LnR5cGUgPT09ICdwYXR0ZXJuJylcbiAgICApXG5cbiAgICBjb25zdCBsaXN0RW50aXRpZXMgPSBhd2FpdCB0aGlzLmVudGl0eUV4dHJhY3Rvci5leHRyYWN0TGlzdHMoXG4gICAgICB0ZXh0LFxuICAgICAgbGFuZyxcbiAgICAgIGN1c3RvbUVudGl0eURlZnMuZmlsdGVyKGVudCA9PiBlbnQudHlwZSA9PT0gJ2xpc3QnKVxuICAgIClcblxuICAgIGNvbnN0IHN5c3RlbUVudGl0aWVzID0gYXdhaXQgdGhpcy5zeXN0ZW1FbnRpdHlFeHRyYWN0b3IuZXh0cmFjdCh0ZXh0LCBsYW5nKVxuICAgIGRlYnVnRW50aXRpZXModGV4dCwgeyBzeXN0ZW1FbnRpdGllcywgcGF0dGVybkVudGl0aWVzLCBsaXN0RW50aXRpZXMgfSlcbiAgICByZXR1cm4gWy4uLnN5c3RlbUVudGl0aWVzLCAuLi5wYXR0ZXJuRW50aXRpZXMsIC4uLmxpc3RFbnRpdGllc11cbiAgfVxuXG4gIHByaXZhdGUgYXN5bmMgX2V4dHJhY3RJbnRlbnRzKFxuICAgIHRleHQ6IHN0cmluZyxcbiAgICBpbmNsdWRlZENvbnRleHRzOiBzdHJpbmdbXVxuICApOiBQcm9taXNlPHsgaW50ZW50czogc2RrLk5MVS5JbnRlbnRbXTsgaW50ZW50OiBzZGsuTkxVLkludGVudDsgaW5jbHVkZWRDb250ZXh0czogc3RyaW5nW10gfT4ge1xuICAgIGNvbnN0IGV4YWN0SW50ZW50ID0gdGhpcy5fZXhhY3RJbnRlbnRNYXRjaGVyLmV4YWN0TWF0Y2goc2FuaXRpemUodGV4dC50b0xvd2VyQ2FzZSgpKSwgaW5jbHVkZWRDb250ZXh0cylcblxuICAgIGlmIChleGFjdEludGVudCkge1xuICAgICAgcmV0dXJuIHtcbiAgICAgICAgaW5jbHVkZWRDb250ZXh0cyxcbiAgICAgICAgaW50ZW50OiBleGFjdEludGVudCxcbiAgICAgICAgaW50ZW50czogW2V4YWN0SW50ZW50XVxuICAgICAgfVxuICAgIH1cblxuICAgIGNvbnN0IGludGVudHMgPSBhd2FpdCB0aGlzLmludGVudENsYXNzaWZpZXIucHJlZGljdCh0ZXh0LCBpbmNsdWRlZENvbnRleHRzKVxuICAgIGNvbnN0IGludGVudCA9IGZpbmRNb3N0Q29uZmlkZW50SW50ZW50TWVhblN0ZChpbnRlbnRzLCB0aGlzLmNvbmZpZGVuY2VUcmVzaG9sZClcbiAgICBpbnRlbnQubWF0Y2hlcyA9IGNyZWF0ZUludGVudE1hdGNoZXIoaW50ZW50Lm5hbWUpXG5cbiAgICAvLyBhbHRlciBjdHggd2l0aCB0aGUgZ2l2ZW4gcHJlZGljdGlvbnMgaW4gY2FzZSB3aGVyZSBubyBjdHggd2VyZSBwcm92aWRlZFxuICAgIGluY2x1ZGVkQ29udGV4dHMgPSBfLmNoYWluKGludGVudHMpXG4gICAgICAubWFwKHAgPT4gcC5jb250ZXh0KVxuICAgICAgLnVuaXEoKVxuICAgICAgLnZhbHVlKClcblxuICAgIGRlYnVnSW50ZW50cyh0ZXh0LCB7IGludGVudHMgfSlcblxuICAgIHJldHVybiB7XG4gICAgICBpbmNsdWRlZENvbnRleHRzLFxuICAgICAgaW50ZW50cyxcbiAgICAgIGludGVudFxuICAgIH1cbiAgfVxuXG4gIHByaXZhdGUgYXN5bmMgX2V4dHJhY3RTbG90cyhcbiAgICB0ZXh0OiBzdHJpbmcsXG4gICAgaW50ZW50OiBzZGsuTkxVLkludGVudCxcbiAgICBlbnRpdGllczogc2RrLk5MVS5FbnRpdHlbXVxuICApOiBQcm9taXNlPHNkay5OTFUuU2xvdHNDb2xsZWN0aW9uPiB7XG4gICAgY29uc3QgaW50ZW50RGVmID0gYXdhaXQgdGhpcy5zdG9yYWdlLmdldEludGVudChpbnRlbnQubmFtZSlcbiAgICByZXR1cm4gYXdhaXQgdGhpcy5zbG90RXh0cmFjdG9yLmV4dHJhY3QodGV4dCwgaW50ZW50RGVmLCBlbnRpdGllcylcbiAgfVxuXG4gIHByaXZhdGUgYXN5bmMgX2V4dHJhY3QodGV4dDogc3RyaW5nLCBpbmNsdWRlZENvbnRleHRzOiBzdHJpbmdbXSk6IFByb21pc2U8c2RrLklPLkV2ZW50VW5kZXJzdGFuZGluZz4ge1xuICAgIGxldCByZXQ6IGFueSA9IHsgZXJyb3JlZDogdHJ1ZSB9XG4gICAgY29uc3QgdDEgPSBEYXRlLm5vdygpXG4gICAgdHJ5IHtcbiAgICAgIHJldC5sYW5ndWFnZSA9IGF3YWl0IHRoaXMubGFuZ0RldGVjdG9yLmlkZW50aWZ5KHRleHQpXG4gICAgICByZXQgPSB7IC4uLnJldCwgLi4uKGF3YWl0IHRoaXMuX2V4dHJhY3RJbnRlbnRzKHRleHQsIGluY2x1ZGVkQ29udGV4dHMpKSB9XG4gICAgICByZXQuZW50aXRpZXMgPSBhd2FpdCB0aGlzLl9leHRyYWN0RW50aXRpZXModGV4dCwgcmV0Lmxhbmd1YWdlKVxuICAgICAgcmV0LnNsb3RzID0gYXdhaXQgdGhpcy5fZXh0cmFjdFNsb3RzKHRleHQsIHJldC5pbnRlbnQsIHJldC5lbnRpdGllcylcbiAgICAgIGRlYnVnRW50aXRpZXMoJ3Nsb3RzJywgeyB0ZXh0LCBzbG90czogcmV0LnNsb3RzIH0pXG4gICAgICByZXQuZXJyb3JlZCA9IGZhbHNlXG4gICAgfSBjYXRjaCAoZXJyb3IpIHtcbiAgICAgIHRoaXMubG9nZ2VyLmF0dGFjaEVycm9yKGVycm9yKS5lcnJvcihgQ291bGQgbm90IGV4dHJhY3Qgd2hvbGUgTkxVIGRhdGEsICR7ZXJyb3J9YClcbiAgICB9IGZpbmFsbHkge1xuICAgICAgcmV0Lm1zID0gRGF0ZS5ub3coKSAtIHQxXG4gICAgICByZXR1cm4gcmV0IGFzIHNkay5JTy5FdmVudFVuZGVyc3RhbmRpbmdcbiAgICB9XG4gIH1cbn1cbiJdfQ==