"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = require("lodash");

var _engine = _interopRequireDefault(require("./engine"));

var _fiveFold = require("./tools/five-fold");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * A specialized version of the NLU Engine that has added support for Confusion Matrix
 * This engine is much slower because it trains multiple models instead of only one
 * For practical reasons, set `computeConfusionOnTrain` to true to enable Confusion Matrix computations.
 */
class ConfusionEngine extends _engine.default {
  constructor(...args) {
    super(...args);

    _defineProperty(this, "modelName", '');

    _defineProperty(this, "modelIdx", 0);

    _defineProperty(this, "originalModelHash", '');

    _defineProperty(this, "computeConfusionOnTrain", false);
  }

  async init() {
    await super.init();
  }

  async trainModels(intentDefs, modelHash) {
    await super.trainModels(intentDefs, modelHash);

    if (!this.computeConfusionOnTrain) {
      return;
    }

    const dataset = this._definitionsToEntry(intentDefs);

    const folder = new _fiveFold.FiveFolder(dataset);
    this.modelIdx = 0;
    this.modelName = '';
    this.originalModelHash = modelHash;
    await folder.fold('intents', this._trainIntents.bind(this), this._evaluateIntents.bind(this));
    await this._processResults(folder.getResults());
  }

  async _processResults(results) {
    const reportUrl = process['EXTERNAL_URL'] + `/api/v1/bots/${this.botId}/mod/nlu/confusion/${this.originalModelHash}`;
    await this.storage.saveConfusionMatrix(this.originalModelHash, results);
    const intents = results['intents'];
    this.logger.debug('=== Confusion Matrix ===');
    this.logger.debug(`F1: ${intents['all'].f1} P1: ${intents['all'].precision} R1: ${intents['all'].recall}`);
    this.logger.debug(`Details available here: ${reportUrl}`);
  }

  _definitionsToEntry(def) {
    return (0, _lodash.flatten)(def.map(x => x.utterances.map(u => ({
      definition: x,
      utterance: u
    }))));
  }

  _entriesToDefinition(entries) {
    const groups = (0, _lodash.groupBy)(entries, x => x.definition.name + '|' + x.definition.contexts.join('+'));
    return Object.keys(groups).map(x => ({ ...groups[x][0].definition,
      utterances: groups[x].map(x => x.utterance)
    }));
  }

  async _trainIntents(dataSet) {
    const defs = this._entriesToDefinition(dataSet);

    this.modelName = `${this.originalModelHash}-fold${this.modelIdx++}`;
    await super.trainModels(defs, this.modelName);
  }

  async _evaluateIntents(dataSet, record) {
    const defs = this._entriesToDefinition(dataSet);

    await this.loadModels(defs, this.originalModelHash);
    const expected = await Promise.mapSeries(dataSet, (__, idx) => this.extract(dataSet[idx].utterance, []));
    await this.loadModels(defs, this.modelName);
    const actual = await Promise.mapSeries(dataSet, (__, idx) => this.extract(dataSet[idx].utterance, []));
    dataSet.forEach((__, idx) => record(expected[idx].intent.name, actual[idx].intent.name));
  }

}

exports.default = ConfusionEngine;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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