"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Testing = void 0;

var _nanoid = _interopRequireDefault(require("nanoid"));

var _path = _interopRequireDefault(require("path"));

var _recorder = require("./recorder");

var _runner = require("./runner");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const SCENARIO_FOLDER = 'scenarios';

class Testing {
  constructor(bp, botId) {
    _defineProperty(this, "bp", void 0);

    _defineProperty(this, "botId", void 0);

    _defineProperty(this, "_recorder", void 0);

    _defineProperty(this, "_runner", void 0);

    _defineProperty(this, "_scenarios", void 0);

    _defineProperty(this, "_interval", void 0);

    this.bp = bp;
    this.botId = botId;
    this._recorder = new _recorder.Recorder();
    this._runner = new _runner.SenarioRunner(bp);
  }

  async startRecording(chatUserId) {
    await this._ensureHooksEnabled();

    this._recorder.startRecording(chatUserId);
  }

  endRecording() {
    return this._recorder.stopRecording();
  }

  getStatus() {
    return {
      recording: this._recorder.isRecording(),
      running: this._runner.isRunning()
    };
  }

  async getScenarios() {
    if (!this._scenarios) {
      await this._loadScenarios();
    }

    return this._scenarios.map(({
      name,
      steps
    }) => {
      return {
        name,
        steps,
        ...this._runner.getStatus(name)
      };
    });
  }

  processIncomingEvent(event) {
    this._recorder.processIncoming(event);

    return this._runner.processIncoming(event);
  }

  processCompletedEvent(event) {
    this._recorder.processCompleted(event);

    this._runner.processCompleted(event);
  }

  async saveScenario(name, scenario) {
    await this.bp.ghost.forBot(this.botId).upsertFile(SCENARIO_FOLDER, name + '.json', JSON.stringify(scenario, undefined, 2));
    await this._loadScenarios();
  }

  _executeScenario(scenario) {
    const eventDestination = {
      channel: 'web',
      botId: this.botId,
      threadId: undefined,
      target: `test_${(0, _nanoid.default)()}`
    };

    this._runner.runScenario({ ...scenario
    }, eventDestination);
  }

  async executeSingle(liteScenario) {
    await this._ensureHooksEnabled();

    this._runner.startReplay(); // TODO perform scenario validation here


    const scenario = await this.bp.ghost.forBot(this.botId).readFileAsObject(SCENARIO_FOLDER, liteScenario.name + '.json');

    this._executeScenario({ ...liteScenario,
      ...scenario
    });
  }

  async executeAll() {
    await this._ensureHooksEnabled();
    const scenarios = await this._loadScenarios();

    this._runner.startReplay();

    scenarios.forEach(scenario => {
      this._executeScenario(scenario);
    });
  }

  async _ensureHooksEnabled() {
    if (!this._interval) {
      this._interval = setInterval(this._waitTestCompletion.bind(this), 2000);
    }

    await this.bp.experimental.enableHook('00_recorder', 'before_incoming_middleware', 'testing');
    await this.bp.experimental.enableHook('00_recorder', 'after_event_processed', 'testing');
  }

  async _waitTestCompletion() {
    if (!this._runner.isRunning() && !this._recorder.isRecording()) {
      await this.bp.experimental.disableHook('00_recorder', 'before_incoming_middleware', 'testing');
      await this.bp.experimental.disableHook('00_recorder', 'after_event_processed', 'testing');
      clearInterval(this._interval);
      this._interval = undefined;
    }
  }

  async _loadScenarios() {
    const files = await this.bp.ghost.forBot(this.botId).directoryListing(SCENARIO_FOLDER, '*.json');
    this._scenarios = await Promise.map(files, async file => {
      const name = _path.default.basename(file, '.json');

      const scenarioSteps = await this.bp.ghost.forBot(this.botId).readFileAsObject(SCENARIO_FOLDER, file);
      return {
        name,
        ...scenarioSteps
      };
    });
    return this._scenarios;
  }

}

exports.Testing = Testing;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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