#!/usr/bin/perl

use strict;
# there isn't a _reliable_ way to dynamically include a module and have
#   it successfully resolve its dependencies
use lib 'include';

use AutomateSchedule::Install;
use File::List;
use Cwd;

umask 022;

my $startingDir = cwd();
my $fileDir = 'files-server';
my $installer = AutomateSchedule::Install->new($fileDir) or die "Unable to load Automate Schedule Installer module.";
my $os = $installer->os();
my $rawArch = $installer->rawArch();
my $arch = $installer->arch();
my $platform = $installer->platform();
my $bitWidth = $installer->bitWidth();
my $osversion = $installer->osVersion();
my $useJavaVersion = $installer->useJavaVersion();
my $postgresqlPossible = $installer->postgresqlPossible();
my $jreVersion;
my $whoami = $installer->whoami();
my $tmpSkybotLog = "/tmp/skybot-$whoami.log";
system ("rm -f $tmpSkybotLog") == 0 || die("Unable to remove temp log $tmpSkybotLog: $!\n");
my $tmpUserLog = $startingDir . "/user-$whoami.log";
system ("rm -f $tmpUserLog") == 0 || die("Unable to remove temp log $tmpUserLog: $!\n");


my $relativeWebinf = "webapps/automate-schedule/WEB-INF";
my $webinf = "$fileDir/$relativeWebinf";
$ENV{'LD_LIBRARY_PATH'} = $ENV{'LD_LIBRARY_PATH'} . ":$webinf/bin/$platform";

print("\nWelcome to Automate Schedule Server, version " . $installer->buildInfo() . "\n");
print("The current user is $whoami, this system is $platform $osversion, $bitWidth bits\n\n");

# Determine which PostgreSQL install file to use, and how to decompress it
my $dbinstallfile;
my $dbinstallcmd;

if ( ($os eq 'AIX') && ($rawArch eq 'powerpc') ) {
    $dbinstallfile = 'postgresql9-AIX-powerpc.tar.Z';
    $dbinstallcmd = 'zcat _FILE_ | tar -xf -';
}

if ( ($os eq 'Linux') &&
    (($rawArch eq 'i686') || ($rawArch eq 'x86_64') || ($rawArch eq 'ppc64')) ) {
    $dbinstallfile = "postgresql-10.3-2-Linux-$rawArch.tgz";
    $dbinstallcmd = 'tar xzf _FILE_';
}

if ( ($os eq 'Linux') && ($rawArch eq 'ppc64le') ) {
    print("This platform/OS version is not supported for the Automate Schedule Server.\n");
    exit 1;
}

if ( ($os eq 'SunOS') && ($arch eq 'sparc64') ) {
    $dbinstallfile = "postgresql9-SunOS-sparc.tar.bz2";
    $dbinstallcmd = 'bzip2 -cd _FILE_ | tar -xf -';
}

if ( !defined($dbinstallfile) || $postgresqlPossible eq 'false') {
    print("This platform/OS version is not supported for the Automate Schedule Server.\n");
    exit 1;
}

if ( $bitWidth eq '32' ) {
    print("The Automate Schedule Server does not support 32-bit operating systems.  Please install on a 64-bit system.\n");
    exit 1; 
}
 
my $automateUser;
my $automateScheduleDir;
my $oldSkybotDir;
my $createAutomateUser = 0;
my $addReplicationPermission = 0;

if (($whoami eq 'root') || ($whoami =~ m/automate/i)) {
    # See if there's a automate user defined
    $automateUser = $installer->automateUser();
    $automateScheduleDir = $installer->automateScheduleDir();

    if (!defined($automateUser) && ($whoami eq 'root') ) {
        print("* The Automate Schedule files and database need to be owned by a non-root user,\n");
        print("  a local 'automate' user will be created.\n");
        $automateUser = 'automate';
        $installer->setAutomateUser($automateUser);
        $automateScheduleDir = '/opt/automate-schedule';
        $oldSkybotDir = '/opt/skybot';
        $createAutomateUser = 1;
        $installer->setAutomateScheduleDir($automateScheduleDir);
        $installer->setUserHome($automateScheduleDir);
    }
} else {
    # install it into the 'automate-schedule' dir in their home dir
    $automateUser = $whoami;
    $automateScheduleDir = $installer->userHome() . "/automate-schedule";
    $oldSkybotDir = $installer->userHome() . "/skybot";
    $installer->setAutomateScheduleDir($automateScheduleDir);
}

my $automateServerDir = "$automateScheduleDir/server";
my $oldSkybotServerDir = "$oldSkybotDir/server";

# This only applies to old Skybot installations.
if (-d "$oldSkybotDir/scheduler") {
    # the previous generation of installers used "scheduler" instead of "server"
    $oldSkybotServerDir = "$oldSkybotDir/scheduler"
}

# If old Skybot is installed this is a rebrand.
my $rebrand = 0;
my $skybotGID;
if ( -e "$oldSkybotServerDir/webapps/skybot-scheduler/WEB-INF/web.xml") {
    $rebrand = 1;
    print("* Skybot Scheduler will be rebranded to Automate Schedule.\n");
}

# see if we're installing or updating.
my $install = 1;

# If this is a rebrand, we are updating.
if ($rebrand == 1) {
	$install = 0;
	checkIfOldServerRunning();
    if (-d "$oldSkybotDir/agent") {
      # make sure the old agent is shut down too
    	my $output = `$oldSkybotDir/agent/stop.sh`;
    }
    
    # Need to reset this temporarily for findJava().
    $installer->setAutomateScheduleDir($oldSkybotDir);
    
} else {
	if ( -e "$automateServerDir/webapps/automate-schedule/WEB-INF/config/entServer.xml") {
   		$install = 0;
	}
	checkIfServerRunning();
	if (-d "$automateScheduleDir/agent") {
   		# make sure the agent is shut down too
   		my $output = `$automateScheduleDir/agent/stop.sh`;
	}
}

# resolve Java
my $javaInstallFile;
my $javaInstallCmd;
my $javaDir = $installer->findJava();
my $jre;

# added to prevent java 1.6 installations.
if ($useJavaVersion == 6) {
	die("Sorry, but this platform ($platform) os ($os) and architecture ($arch) require an unsupported java version.");
}

if ($rebrand == 1) {
	# Set this back after findJava() call.
	$installer->setAutomateScheduleDir($automateScheduleDir);
}

if (defined($javaDir)) {
    print("* Using Java at $javaDir\n");
    expandWarfile();
    $installer->hasNativesz("$fileDir/webapps/automate-schedule/WEB-INF/bin/") || 
        die("This platform ($platform) is not supported.\n");

    testNativesz();
    
} else {
    if ($install == 0) {
        # there's an existing installation of Automate Schedule, but we don't know
        #   where to find Java. This means we can't read the existing
        #   configuration, because that requires Java.
        die("The existing Automate Schedule Server configuration could not be read, because the\n   JAVA_HOME environment does not point to a usable copy of Java 1.7 or higher.\n");
    }

    $jre = "jre1$useJavaVersion";
    $jreVersion = "$useJavaVersion";

    if ( ($os eq 'Linux') && ($arch eq 'i686') ) {
            $javaInstallFile = "$jre-$os-$arch.tgz";
            $javaInstallCmd = 'tar xzf _FILE_';
    }

    if ( ($os eq 'Linux') &&
        (($arch eq 'x86_64') || ($arch eq 'ppc64') || ($arch eq 'ppc64le'))  ) {
            $javaInstallFile = "${jre}_161-$os-$arch.tgz";
            $javaInstallCmd = 'tar xzf _FILE_';
    }

    if ( ($os eq 'AIX') && ($rawArch eq 'powerpc') ) {
        $javaInstallFile = "$jre-$os-$arch.tar.Z";
        $javaInstallCmd = 'zcat _FILE_ | tar -xf -';
    }


    if (!defined($javaInstallFile)) {
        # we don't have an installer for this combination.
        print("* Unable to find a usable Java Runtime Environment (JRE).\n");
        print("  Please download and install Java version 1.7 or 1.8 and point the environment\n");
        print("      variable JAVA_HOME at the directory.\n");
        print("  Visit http://www.helpsystems.com/getjava for more info.\n");
        exit 1;
    }

    $javaDir = "$automateScheduleDir/jre";
    print ("* Java $jreVersion will be installed at $javaDir\n");
}

# make sure this is set correctly
$ENV{'JAVA_HOME'} = $javaDir;

my $httpPort;
my $protocol;
my $dbPort;
my $tlPort;
my $ajpPort;
my $shutdownPort;
my $holdAgentsFlag = 'false';
my $holdSchedulerFlag = 'false';
my $needToInstallAgent = 0;
my $dbVersion;

my $currentProductServerDir = $automateServerDir;
if ($rebrand == 1) {
    $currentProductServerDir = $oldSkybotServerDir;
}

# see if we're installing or updating.
if ($install == 0) {
    # looks like an UPDATE
    print("* The Automate Schedule Server will be updated at $currentProductServerDir\n");
    my $configSuccess;
    if ($rebrand == 1) {
        $configSuccess = system("$currentProductServerDir/configServer.sh -s$currentProductServerDir,-g$startingDir > config.log");
    } else {
        $configSuccess = system("$fileDir/configServer.sh -s$currentProductServerDir,-g$startingDir > config.log");
    }
    if ($configSuccess != 0) {
        system("cat config.log");
        system("rm -f config.log");
        die("The existing Automate Schedule Server configuration could not be read.\n");
    }
    if (! -r 'setupinfo.txt') {
        system("cat config.log");
        system("rm -f config.log");
        die("Unable to read the existing configuration.\n");
    }
    system("rm -f config.log");

    my $existingConfig = $installer->readFileHash('setupinfo.txt');
    $httpPort = $existingConfig->{'EXISTING_HTTP_PORT_NUMBER'};
    $protocol = $existingConfig->{'EXISTING_PROTOCOL'};
    $dbPort = $existingConfig->{'EXISTING_POSTGRES_PORT_NUMBER'};
    $tlPort = $existingConfig->{'EXISTING_SERVER_PORT_NUMBER'};
    $shutdownPort = $existingConfig->{'EXISTING_SHUTDOWN_PORT_NUMBER'};
    $ajpPort = $existingConfig->{'EXISTING_AJP13_PORT_NUMBER'};
    $holdAgentsFlag = $existingConfig->{'EXISTING_HOLD_AGENTS_FLAG'};
    $holdSchedulerFlag = $existingConfig->{'EXISTING_HOLD_SCHEDULER_FLAG'};
} else {
    print("* The Automate Schedule Server will be installed at $automateServerDir\n");
    $httpPort = $installer->findOpenPort(8008);
    $protocol = "http";
    $dbPort = $installer->findOpenPort(7432);
    $tlPort = $installer->findOpenPort(7472);
    $ajpPort = $installer->findOpenPort(8009);
    $shutdownPort = $installer->findOpenPort(8005);
}

$holdAgentsFlag = $installer->pickValue($holdAgentsFlag, 'false');
$holdSchedulerFlag = $installer->pickValue($holdSchedulerFlag, 'false');

print("* Using port $httpPort for web browser connections\n");
print("* Using port $tlPort for agent connections\n");
print("* Using port $ajpPort for AJP connections\n");
print("* Using port $shutdownPort for shutdown notification\n");
print("* Using port $dbPort for the database\n");

my $loop = 1;
while ($loop) {
    print("Press 'Enter' to continue with the installation,\n");
    print("  type 'c' and press 'Enter' to change the settings,\n");
    print("  or press Control-C to exit.\n");
    print("Begin installation? ");
    my $input = <STDIN>;
    chomp($input);
    if ($input eq '' || $input eq 'y' || $input eq 'Y') {
        $loop = 0;
    }
    elsif ($input eq 'c' || $input eq 'C' ) {
        manualConfig();
    }
}

## JEL Changed.  If we are not rebranding, create the product user here.
if ($rebrand == 0) {
    createProductUser();
} else {
    # If we are rebranding, the product user will be created in the sub.
    moveProductFolder();
    
    # Set this here so it is correct when configAgentScripts is called.
    $javaDir = "$automateScheduleDir/jre";
    
    # Once we move the product, update the env var JAVA_HOME
    $ENV{'JAVA_HOME'} = $javaDir;
}

# resolve the necessary downloads
my $sapFile = "libsapjco3.so";
my $sapLibPath = "bin/$platform/$sapFile";
my @remainingDownloads = ($dbinstallfile, $sapLibPath);

if (defined($javaInstallFile)) {
    installJava();

    expandWarfile();
    $installer->hasNativesz("$fileDir/webapps/automate-schedule/WEB-INF/bin/") || 
        die("Sorry, but this platform ($platform) is not supported.\n");

    # if JRE7 was put down, run 'execstack -c <libfile>'
    if ( $jreVersion eq '7') {
        # debug lines
        # print("path= files-server/webapps/automate-schedule/WEB-INF/bin/Linux/$rawArch/libnativeSz.so\n");
        # print("execstack -c $fileDir/webapps/automate-schedule/WEB-INF/bin/Linux/$rawArch/libnativeSz.so\n");
        my $result = system("execstack -c files-server/webapps/automate-schedule/WEB-INF/bin/Linux/$rawArch/libnativeSz.so");
        if ($result > 0) {
           print("Warning only - Unable to mark nativeSz as not requiring an executable stack\n");
        }
    }
    testNativesz();
}
shift(@remainingDownloads);


$installer->makeDir($automateServerDir);
my $dbDataDir = "$automateServerDir/pgsql/data";
$ENV{'LIBPATH'} = $ENV{'LIBPATH'} . ":$automateServerDir/pgsql/lib";
$ENV{'LD_LIBRARY_PATH'} = $ENV{'LD_LIBRARY_PATH'} . ":$automateServerDir/pgsql/lib";

if (! -e "$dbDataDir/PG_VERSION") {
    installDatabaseFiles();
    initializeDatabase();
} else {
    # see what version of Postgres is installed
    chomp($dbVersion = `cat $dbDataDir/PG_VERSION`);
    if (($os eq 'AIX' && $dbVersion < 9.1) || ( $os ne 'AIX' && $dbVersion < 10 )) {
        print("Preparing to update the Automate Schedule Database Server\n");
        my $newDBdir = "$automateServerDir/pgsql";
        my $now = time();
        my $oldDBdir = "$automateServerDir/pgsql.$now";
        system("mv $newDBdir $oldDBdir") == 0 || die("Unable to rename $newDBdir to $oldDBdir\n");
        installDatabaseFiles();

        eval {
            migrateDatabase($oldDBdir);
        };
        if ($@) {
            my $err = $@;
            # delete the new postgress dir, and restore the old one.
            system("rm -rf $newDBdir");
            system("mv -f $oldDBdir $newDBdir");
            die("$err\n");
        }

        chdir $startingDir;
    }
}
shift(@remainingDownloads);

downloadExtraBinaries();
copyFiles();
configDbScripts();
configServerScripts();
configInfo();

if ($addReplicationPermission == 1) {
    addReplicationPermission();
}

#downloadAgents();
copyAgents();
$needToInstallAgent = 1;


## JEL Added sub for creating user automate so it can be called earlier for
## a rebranding.
sub createProductUser {
	my $fixDirPermissions = 0;
	if ($createAutomateUser) {
    	system("/usr/sbin/useradd -d $automateScheduleDir automate") == 0 || die("Unable to create 'automate' user.\n");
    	$fixDirPermissions = 1;
	}

	$skybotGID = $installer->getGID();

	# Time to start installing files to the destination dir.
	if ( ! -e $automateScheduleDir) {
    	$installer->makeDir($automateScheduleDir);
    	$fixDirPermissions = 1;
	}

	if ($fixDirPermissions) {
    	system("chmod 755 $automateScheduleDir") == 0 || die("Unable to set permissions on $automateScheduleDir\n");
	}
}

## JEL Added sub for moving product folder and creating product user when rebranding.
sub moveProductFolder {
    
    # First rename $oldSkybotDir/scheduler to $oldSkybotDir/server if needed.
    if (-d "$oldSkybotDir/scheduler") {
    	system("mv $oldSkybotDir/scheduler $oldSkybotDir/server") == 0 || die("Unable to rename $oldSkybotDir/scheduler to $oldSkybotDir/server\n");
    }
    
    # Before moving, remove these links as they will become broken and will be 
    # recreated at runtime as needed.
    system("rm -f $oldSkybotServerDir/schent-mp-module.jar > /dev/null 2>&1");
    system("rm -f $oldSkybotServerDir/transport-module.jar > /dev/null 2>&1");

    # Move to the rebranded folder if already installed.
    #     (skybot folder -> automate-schedule folder)
    print("Moving $oldSkybotDir to $automateScheduleDir.\n");
    system("mv $oldSkybotDir $automateScheduleDir") == 0 || die("Unable to rename $oldSkybotDir to $automateScheduleDir\n");
    createProductUser();
    if ($whoami eq 'root') {
        system("chown -R $automateUser:$skybotGID $automateScheduleDir") == 0 || die("Unable to change owner of $automateScheduleDir\n");
        system("/usr/sbin/userdel skybot") == 0 || print("Unable to delete 'skybot' user.\n");
    }
    
    # Move webapps/skybot-scheduler to webapps/automate-schedule
    print("Moving $automateServerDir/webapps/skybot-scheduler to $automateServerDir/webapps/automate-schedule.\n");
    system("mv $automateServerDir/webapps/skybot-scheduler $automateServerDir/webapps/automate-schedule") == 0 || die("Unable to rename $automateServerDir/webapps/skybot-scheduler to $automateServerDir/webapps/automate-schedule\n");

}

sub installDatabaseFiles { 

    $installer->maybeDownload($dbinstallfile, \@remainingDownloads) || die("\n");
    print("Installing the Automate Schedule Database at $automateServerDir/pgsql\n");
    chdir "$automateServerDir" or die("Unable to change dir to $automateServerDir\n");
    $dbinstallfile= '"' . "$startingDir/$dbinstallfile" . '"';
    $dbinstallcmd =~ s/_FILE_/$dbinstallfile/g;
    print("Extracting database files: $dbinstallcmd\n");
    system("$dbinstallcmd") == 0 || die("Unable to extract database files.\n");
        
    my $pwfile = "$startingDir/$fileDir/pgtemp.txt";
    if ($whoami eq 'root') {
        system("chown -R $automateUser:$skybotGID .") == 0 || die("Unable to switch ownership of Automate Schedule DB files to $automateUser\n");
        system("chmod 644 $pwfile") == 0 || die("Unable to change permissions on $pwfile: $!\n");
    }

    if (! -r $pwfile) {
        die("Unable to read DB init file $pwfile\n");
    }

    chdir $startingDir;
    my $userHome = $installer->userHome();
    my $effectiveUsername = $whoami;
    if ($whoami eq 'root') {
        $effectiveUsername = $automateUser;
        $userHome = $installer->automateScheduleHomeDir();
    }
    system("cp $fileDir/pgpass.txt $userHome/.pgpass") == 0 ||
        die("Unable to move DB password file to $userHome\n");
    system("chmod 0600 $userHome/.pgpass") == 0 || die("Unable to secure the DB password file in $userHome\n");
    system("cp $pwfile $userHome/pgtemp.txt") == 0 || die("Unable to move initdb password file to $userHome\n");
    if ($whoami eq 'root') {
        system("chown $automateUser:$skybotGID $userHome/.pgpass") == 0 || die("Unable to change the ownership of $userHome/.pgpass to $automateUser\n");
        system("chown $automateUser:$skybotGID $userHome/pgtemp.txt") == 0 || die("Unable to change the ownership of $userHome/pgtemp.txt to $automateUser\n");
    }
}

sub initializeDatabase { 
    my $dbsettings = shift;
    my $lc_collate;
    my $lc_ctype;
    my $lcFlags = "";

    if (defined($dbsettings)) {
        $lc_collate=$dbsettings->{'lc_collate'};
        $lc_ctype=$dbsettings->{'lc_ctype'};
        $lcFlags = " --lc-collate=$lc_collate --lc-ctype=$lc_ctype";
        print("Initializing the database using flags $lcFlags\n");
    } else {
        print("Initializing the database...\n");
    }

    chdir "$automateServerDir/pgsql" or die("Unable to switch to directory $automateServerDir/pgsql\n");
    
    $ENV{'LANG'} = '';
    $ENV{'PATH'} = $ENV{'PATH'} .":bin";

    my $userHome = $installer->userHome();
    if ($whoami eq 'root') {
        $userHome = $installer->automateScheduleHomeDir();
    }
    my $pwfile = "$userHome/pgtemp.txt";
    my $initCommand = "bin/initdb -A md5 -U Skybot --pwfile $pwfile --encoding=UTF8 $lcFlags -D data > dbinit.log 2>&1";

    if ($whoami eq 'root') {
	    $initCommand = "su $automateUser -c \"$initCommand \"";
    }

    my $success = system($initCommand);
    system("rm $pwfile") == 0 || die("Unable to remove initdb password file in $userHome\n");
    if ($success == -1) {
        die("Unable to start the database initialization: $!\n");
    } elsif ($success != 0) {
        system("cat dbinit.log");
        die("Unable to initialize the database\n");
    }

    my $pgconfig = "data/postgresql.conf";
    my $pgconfigbak = $pgconfig . '.bak';
    # TODO - we may need to enable listen_addresses='*'
    #system("mv $pgconfig $pgconfigbak") == 0 || 
    #    die("Unable to rename $pgconfig to $pgconfigbak\n");
    my $searchText = "^#listen_addresses";
    my $replaceText = "listen_addresses";
    #$installer->copyScript($pgconfigbak, $pgconfig, $searchText, $replaceText) or die("\n");

    system("mv $pgconfig $pgconfigbak") == 0 || 
        die("Unable to rename $pgconfig to $pgconfigbak\n");
    $searchText = "^max_connections = 100";
    $replaceText = "max_connections = 130";
    $installer->copyScript($pgconfigbak, $pgconfig, $searchText, $replaceText) or die("\n");

    system("mv $pgconfig $pgconfigbak") == 0 || 
        die("Unable to rename $pgconfig to $pgconfigbak\n");
    system("grep -v bytea_output $pgconfigbak | grep -v standard_conforming_strings > $pgconfig") == 0 ||
        die("Unable to prepare file $pgconfig");
    system("echo bytea_output = escape >> $pgconfig") == 0 ||
        die("Unable to add 'bytea_output' to $pgconfig");
    system("echo standard_conforming_strings = off >> $pgconfig") == 0 ||
        die("Unable to add 'standard_conforming_strings' to $pgconfig");
    
    if ( ($os eq 'Linux') && ($bitWidth eq '64') ) {
        # trim the shared mem size to 16MB
        system("mv $pgconfig $pgconfigbak") == 0 || 
            die("Unable to rename $pgconfig to $pgconfigbak\n");
        $searchText = "^shared_buffers = 24MB";
        $replaceText = "shared_buffers = 16MB";
        $installer->copyScript($pgconfigbak, $pgconfig, $searchText, $replaceText) or die("\n");
    }

    # append replication settings to postgresql.conf
    open(CONF,">>data/postgresql.conf") or die("Unable to write to postgresql.conf");
    if( $os eq "AIX") {
        print CONF "wal_level = hot_standby\n";
        print CONF "max_wal_senders = 5\n";
    } else {
        print CONF "wal_level = replica\n";
        print CONF "max_wal_senders = 10\n";
    }
    
    print CONF "wal_keep_segments = 32\n";
    print CONF "hot_standby = on\n";
    if ($install == 0) {
        print CONF "port=$dbPort";
    }
    close(CONF);

    open(HBA, ">>data/pg_hba.conf") or die("Unable to write to pg_hba.conf");
    print HBA "host     replication     Skybot      0.0.0.0/0            md5\n";
    close(HBA);
    
    chdir $startingDir;
    
}

# migrates the data from the old database into the new database. Dies if
# something goes wrong.
sub migrateDatabase {
    my $oldDBdir = shift;

    if (!defined($oldDBdir)) {
        die("Migrate Database: old database dir-name was not provided\n");
    }
    
    chdir $oldDBdir || die("Unable to change directory to $oldDBdir: $!\n");

    my $settingsLog = "settings.txt";
    system("rm -f $settingsLog") == 0 || die("Unable to delete file $settingsLog: $!\n");
    
    my $oldHBA = "$oldDBdir/data/pg_hba.conf";
    system("mv -f $oldHBA $oldHBA.bak") == 0 || die("Unable to backup the file $oldHBA\n");

    my $dbStartCommand = "bin/pg_ctl start -D data > $tmpSkybotLog 2>&1";
    if ($whoami eq 'root') {
        $dbStartCommand = "su $automateUser -c \"$dbStartCommand\"";
    }

    my $unixSocket = "/tmp/.s.PGSQL.$dbPort";
    my $dbQuery = "bin/psql -p $dbPort -U Skybot -q -c \"select name, setting from pg_settings\" -o $settingsLog -A -F = postgres";
    if ($whoami eq 'root') {
        $dbQuery =~ s/\"/\\\"/g;
        $dbQuery = "su $automateUser -c \"$dbQuery\"";
    }

    my $dbReady = 0;
    my $retryCount = 10;
    my $newDBdir = "$automateServerDir/pgsql";
    my $newHBA = "$newDBdir/data/pg_hba.conf";
    
    eval {
        open(THEFILE, ">$oldHBA") || die("Unable to write to $oldHBA: $!\n");
        print THEFILE "local all all trust\n";
        close(THEFILE);

        system($dbStartCommand) == 0 || die("Unable to start existing Automate Schedule database server\n" . dumpTempLog($tmpSkybotLog));

        while (!$dbReady) {
            if ( -e $unixSocket) {
                my $status = system("$dbQuery");
                if ( ($status == 0) && -e $settingsLog) {
                    $dbReady = 1;
                }
            } 
            
            if (!$dbReady) {
                sleep 1;
                $retryCount = $retryCount - 1;
                if ($retryCount < 0) {
                    die("Timeout waiting for the existing Automate Schedule database server.\n" . dumpTempLog($tmpSkybotLog));
                }
            }
        }
        
        my $dbsettings = $installer->readFileHash($settingsLog);

        initializeDatabase($dbsettings);
        
        system("mv -f $newHBA $newHBA.bak") == 0 || die("Unable to backup the file $newHBA\n");
        open(THEFILE, ">$newHBA") || die("Unable to write to $newHBA\n");
        print THEFILE "local all all trust\n";
        close(THEFILE);

        my $suCommand;
        if ($whoami eq 'root') {
            $suCommand="su $automateUser -c \"";
        }

        print("Updating database from $dbVersion to 10.3\n");
        if ($os eq 'Linux') {
            my $oldarch="";
            chomp($oldarch = `objdump -f $oldDBdir/bin/pg_ctl | grep architecture:`);
            my $newarch="";
            chomp($newarch = `objdump -f $newDBdir/bin/pg_ctl | grep architecture:`);

            if ( $oldarch ne $newarch ) {
                die("The existing PostgreSQL database architecture is incompatible.\nExisting: $oldarch\nNew: $newarch\n");
            }
        }

        my $dumpCommand = "$oldDBdir/bin/pg_dumpall --port=$dbPort --username=Skybot -w --verbose --database=schent2 --file=\"$automateServerDir/db-upgrade.tmp\" &> dbmigrate.log";

        if (defined($suCommand)) {
            $dumpCommand = "$suCommand $dumpCommand \"";
        }

        chdir "$automateServerDir" or die("Unable to change dir to $automateServerDir\n");

        print("Starting dump of legacy database\n");
        my $success = system($dumpCommand);
        if ($success == 0) {
            $addReplicationPermission = 1;
        } elsif ($success == -1) {
            die("Unable to start the database dump: $!\n");
        } else {
            if ( -e "dbmigrate.log" ) {
                system("cat dbmigrate.log");
            }
            die("Unable to dump the old Automate Schedule database\n");
        }

        my $dbEndCommand = "$oldDBdir/bin/pg_ctl stop -D $oldDBdir/data > $tmpSkybotLog 2>&1";
        if ($whoami eq 'root') {
            $dbEndCommand = "su $automateUser -c \"$dbEndCommand\"";
        }
        system($dbEndCommand) == 0 || die("Unable to stop existing Automate Schedule database server\n" . dumpTempLog($tmpSkybotLog));

        chdir "$newDBdir" || die("Unable to change directory to $newDBdir: $!\n");
        system($dbStartCommand) == 0 || die("Unable to start new Automate Schedule database server\n" . dumpTempLog($tmpSkybotLog));
        $dbReady = 0;
        $retryCount = 10;
        
        while (!$dbReady) {
            if ( -e $unixSocket) {
                my $status = system("$dbQuery ");
                if ( ($status == 0) && -e $settingsLog) {
                    $dbReady = 1;
                }
            } 
            
            if (!$dbReady) {
                sleep 1;
                $retryCount = $retryCount - 1;
                
                if ($retryCount < 0) {
                    die("Timeout waiting for the new Automate Schedule database server.\n" . dumpTempLog($tmpSkybotLog));
                }
            }
        }

        my $dbCreateCmd = "$newDBdir/bin/createdb -U Skybot --port=$dbPort schent2 > $tmpSkybotLog 2>&1";
        if ($whoami eq 'root') {
            $dbCreateCmd = "su $automateUser -c \"$dbCreateCmd\"";
        }
        system($dbCreateCmd) == 0 || die("Unable to create new Automate Schedule database\n" . dumpTempLog($tmpSkybotLog));
        
        print("Starting restore of user data\n");
        my $restoreCommand = "$newDBdir/bin/psql --port=$dbPort --username=Skybot -w --file=\"$automateServerDir/db-upgrade.tmp\" schent2  >> dbmigrate.log 2>&1";

        if (defined($suCommand)) {
            $restoreCommand = "$suCommand $restoreCommand \"";
        }

        chdir "$automateServerDir" or die("Unable to change dir to $automateServerDir\n");

        my $success = system($restoreCommand);
        if ($success == 0) {
            print("Restore of user database successful.");
        } elsif ($success == -1) {
            die("Unable to start the database restore: $!\n");
        } else {
            if ( -e "dbmigrate.log" ) {
                system("cat dbmigrate.log");
            }
            die("Unable to restore the old Automate Schedule database\n");
        }

        my $dbEndCommand = "$newDBdir/bin/pg_ctl stop -D $newDBdir/data > $tmpSkybotLog 2>&1";
        if ($whoami eq 'root') {
            $dbEndCommand = "su $automateUser -c \"$dbEndCommand\"";
        }
        system($dbEndCommand) == 0 || die("Unable to stop new Automate Schedule database server\n" . dumpTempLog($tmpSkybotLog));

        system("rm -f $automateServerDir/db-upgrade.tmp") == 0 || die("Unable to clean up temporary file $automateServerDir/db-upgrade.tmp: $!\n");
        system("mv -f $newHBA.bak $newHBA") == 0 || die("Unable to restore the file $newHBA: $!\n");
        system("mv -f $oldHBA.bak $oldHBA") == 0 || die("Unable to restore the file $oldHBA: $!\n");
        system("rm -rf $oldDBdir") == 0 || die("Unable to clean up temporary directory $oldDBdir : $!\n");
    };

    if ($@) {
        my $err = $@;

        system("mv -f $oldHBA.bak $oldHBA") == 0 || print("Unable to restore the file $oldHBA: $!\n");
        die($err);
    }

    chdir $startingDir;
}

sub addReplicationPermission {

    # start the new/migrated database 
    my $newDBdir = "$automateServerDir/pgsql";
    chdir $newDBdir || die("Unable to change directory to $newDBdir: $!\n");
    my $dbStartCommand = "bin/pg_ctl start -D data > $tmpSkybotLog 2>&1";
    if ($whoami eq 'root') {
        $dbStartCommand = "su $automateUser -c \"$dbStartCommand\"";
    }
    system($dbStartCommand) == 0 || die("Unable to start existing Automate Schedule database server\n" . dumpTempLog($tmpSkybotLog));

    my $unixSocket = "/tmp/.s.PGSQL.$dbPort";

    my $dbReady = 0;
    my $retryCount = 10;
    
        
    $dbReady = 0;
    $retryCount = 10;
    while (!$dbReady) {
        if ( -e $unixSocket) {
            $dbReady = 1;
        } 
            
        if (!$dbReady) {
            sleep 1;
            $retryCount = $retryCount - 1;
            if ($retryCount < 0) {
                die("Timeout waiting for the updated Automate Schedule database server.\n");
            }
        }
    }

    # add the replication role for the Automate Schedule user
    my $quotedString = "'" . 'ALTER USER "Skybot" WITH REPLICATION' . "'";
    my $dbQuery = "bin/psql -A -p $dbPort -U Skybot -q -c $quotedString -o dbpriv.log schent2";
    if ($whoami eq 'root') {
        $dbQuery =~ s/\"/\\\"/g;
        $dbQuery = "su $automateUser -c \"$dbQuery\"";
    }
    my $status = system("$dbQuery");
    if ($status != 0) {
        print("Unable to add the 'DB replication' role to Skybot Schedule.\n");
        print("   This database will not permit live-replication to another server.\n");
    }


    my $dbEndCommand= "bin/pg_ctl stop -D data > $tmpSkybotLog 2>&1";
    if ($whoami eq 'root') {
        $dbEndCommand= "su $automateUser -c \"$dbEndCommand\"";
    }
    system($dbEndCommand) == 0 || die("Unable to stop migrated Automate Schedule database server\n" . dumpTempLog($tmpSkybotLog));
    
    chdir $startingDir;
}

sub configDbScripts {
    # update the db startup/shutdown scripts with the data dir.
    my $dbDataDir = "$automateServerDir/pgsql/data";
    my $scriptName = "startdb.sh";
    my $searchText = "/opt/automate-schedule/server/pgsql/data";
    $installer->copyScript("$fileDir/$scriptName", "$automateServerDir/$scriptName", $searchText, $dbDataDir) or die("\n");

    $scriptName = "stopdb.sh";
    $installer->copyScript("$fileDir/$scriptName", "$automateServerDir/$scriptName", $searchText, $dbDataDir) or die("\n");
}

sub configServerScripts {
    my $scriptName = "startscheduler.sh";
    my $searchText = "=XXX";
    $installer->copyScript("$fileDir/$scriptName", "$automateServerDir/$scriptName", $searchText, "=$javaDir") or die("\n");

    my $scriptName = "stopscheduler.sh";
    $installer->copyScript("$fileDir/$scriptName", "$automateServerDir/$scriptName", $searchText, "=$javaDir") or die("\n");
    
    my $scriptName = "run_java.sh";
    $installer->copyScript("$fileDir/$scriptName", "$automateServerDir/$scriptName", $searchText, "=$javaDir") or die("\n");     
}

sub configInfo {
    chdir $automateServerDir || die("Unable to change directory to $automateServerDir\n");
	
    $holdSchedulerFlag = $installer->numericTrueFalse($holdSchedulerFlag);
    $holdAgentsFlag = $installer->numericTrueFalse($holdAgentsFlag);
    system("./configServer.sh -s$automateServerDir,-rS$holdSchedulerFlag,-rA$holdAgentsFlag,-pES$tlPort,-pDB$dbPort,-pCP$httpPort,-pAJ$ajpPort,-pSD$shutdownPort > $tmpUserLog 2>&1" ) == 0 ||
            die("Unable to configure the server ports\n" . dumpTempLog($tmpUserLog));

    my $config_file = "$automateServerDir/webapps/automate-schedule/WEB-INF/config/entServer.xml";
    system("chmod 644 ${config_file}") == 0 || die("Unable to change file permissions on ${config_file}: $!\n");

    my $tomcat_cfg_file = "$automateServerDir/conf/server.xml";
    system("chmod 644 ${tomcat_cfg_file}") == 0 || die("Unable to change file permissions on ${tomcat_cfg_file}: $!\n");

	system("./configDbServer.sh \"$install\" \"$automateServerDir\"  > $tmpUserLog 2>&1") == 0 ||
        die("Unable to configure the database setup\n" . dumpTempLog($tmpUserLog));
    chdir $startingDir;
}


sub installJava {

    $installer->maybeDownload($javaInstallFile, \@remainingDownloads) || die("\n");

    $installer->makeDir("$automateScheduleDir");
    chdir "$automateScheduleDir" or die("Unable to change dir to $automateScheduleDir\n");
    $javaInstallFile = '"' . "$startingDir/$javaInstallFile" . '"';
    $javaInstallCmd =~ s/_FILE_/$javaInstallFile/g;
    print("Executing Java install command: $javaInstallCmd\n");
    system($javaInstallCmd) == 0 || die("Java install failed.\n");
    if ($whoami eq 'root') {
        system("chown -R $automateUser:$skybotGID .") == 0 || die("Unable to switch ownership of Automate Schedule Java files to $automateUser\n");
    }
    chdir $startingDir;
}

sub expandWarfile {

    print("* Preparing the installer...\n");
    my $warDir = "$fileDir/webapps/automate-schedule";
    system("sh -c \"rm -rf $warDir/*\"") == 0 || die("Unable to purge directory $fileDir: $!\n");

    chdir $warDir or die("Unable to change directory to $warDir: $!\n");
    my $line = `ls -ald .`;
    chomp($line);
    my @pieces = split / +/,$line;
    my $owner = @pieces[2];

    system("$javaDir/bin/java sun.tools.jar.Main -xf ../automate-schedule.war")
        == 0 || die("Unable to extract installer files to $warDir: $!\n");
    if ("$whoami" ne $owner) {
        system("chown -R $owner:$skybotGID .") == 0 || die(
            "Unable to change ownership of $warDir to $owner: $!\n");
    }

    chdir $startingDir;
    return 1;
}

sub testNativesz {
    my $javaExec = "$javaDir/bin/java";
    my $javaFlags = '';
    if ($arch eq 'sparc64') {
        $javaFlags = '-d64';
    }
    my $libs = new File::List("$webinf/lib");
    my @files = @{ $libs->find("\.jar\$") };
    my $classpath;
    for (@files) {
        $classpath = $classpath . ":$_";
    }
    # debug lines
    #    print("Here are the details about my JVMTests call from installServer.pl::testNativesz : \n");
    #    print("fileDir = $fileDir\n");
    #    print("javaExec = $javaExec\n");
    #    print("javaFlags = $javaFlags\n");
    #    print("LD_LIBRARY_PATH = $ENV{LD_LIBRARY_PATH}\n");
    #    print("classpath = $classpath\n");
    #    print("\n");
    my $exitCode = system("$javaExec $javaFlags -cp $classpath com.helpsystems.enterprise.install.JVMTests -n");
    if ($exitCode > 0) {
        die("This usually occurs when the bit-width of Java does not match the kernel\n   (such as using 32-bit Java on a 64-bit system)\n");
   }
}

sub downloadExtraBinaries {

    my $localSAPLib = "$startingDir/$sapFile";
    
    my $serverSAPLib = "$webinf/$sapLibPath";
    if ( -f $serverSAPLib ) {
        # good to go
        return;
    }

    if ( ! -f $localSAPLib ) {
        # download it
        $installer->maybeDownload($sapLibPath) || die("\n");
    }

    system("cp $localSAPLib $serverSAPLib") == 0 || die(
        "Unable to copy the SAP library $localSAPLib to the installer dir");

}


sub copyFiles {

    print("Copying product files...\n");

    #preserve existing server config files.
    my @configFiles = qw( conf/server.xml );
    push (@configFiles, "$relativeWebinf/config/entServer.xml");
    my $archiveCommand = "tar -cf existingConfig.tar";
    my $restoreCommand;
    my $realFileList = "";
    foreach my $aFile (@configFiles) {
        if (-e "$automateServerDir/$aFile") {
            $realFileList = $realFileList . " $aFile";
        }
    }
    if (length($realFileList) > 0) {
        chdir($automateServerDir);
        system("$archiveCommand $realFileList") == 0 || 
            die("Unable to preserve existing config files.\n");
        chdir($startingDir);
        $restoreCommand = "tar -xf existingConfig.tar";
    }


    if( $os eq "AIX") {
        my $search = new File::List("$automateServerDir/webapps/automate-schedule/WEB-INF/bin/$platform");
        my @files = @{ $search->find("libnative") };
        my $filecount = @files;
        if ($filecount > 0) {
            # Bug 13913: flush loaded shared libraries on AIX
            #   or not - /usr/sbin/slibclean can only be run as root.
            if ($whoami eq 'root') {
                system("/usr/sbin/slibclean");
            }

            # Otheriwse, a "well known" AIX "workaround" is to just rename it.
            my $result = system("mv -f @files[0] $automateServerDir/webapps/automate-schedule/WEB-INF/bin/$platform/trash");
            if ($result > 0) {
                die("Unable to remove @files[0]\n");
            }
        }
    }

    my @webinf_dirs_to_delete = ("help", "WEB-INF/app", "WEB-INF/bin", "WEB-INF/db", "WEB-INF/gems", "WEB-INF/lib", "WEB-INF/vendor",
        "WEB-INF/config/environments", "WEB-INF/config/initializers", "WEB-INF/config/locales");
    foreach my $deleteDir (@webinf_dirs_to_delete) {
        system("rm -rf $automateServerDir/webapps/automate-schedule/$deleteDir > /dev/null 2>&1");
    }

    system("rm -f $startingDir/$fileDir/schent-mp-module.jar > /dev/null 2>&1");
    system("rm -f $startingDir/$fileDir/transport-module.jar > /dev/null 2>&1");
    
    #remove obsolete/upgraded files from server install
    system("rm -f $automateServerDir/webapps/ROOT/WEB-INF/lib/commons-httpclient-3.1.jar > /dev/null 2>&1");
    

    system("cp -R -p $startingDir/$fileDir/* $automateServerDir") == 0 || die("Unable to copy product files to $automateServerDir\n");
    system("rm -f $automateServerDir/pg*.txt");

    # DE14060
    # if we are on AIX, We need to create a symbolic link for the sigar library for AIX 7.1
    if ($os eq 'AIX') {
        print("Creating symbolic link for: $automateServerDir/webapps/automate-schedule/WEB-INF/bin/AIX/powerpc64/libsigar-ppc64-aix-7.so\n");
        my $sigar_lnk_result = system("ln -sf $automateServerDir/webapps/automate-schedule/WEB-INF/bin/AIX/powerpc64/libsigar-ppc64-aix-5.so $automateServerDir/webapps/automate-schedule/WEB-INF/bin/AIX/powerpc64/libsigar-ppc64-aix-7.so");
        if ($sigar_lnk_result > 0) {
           print("Error - Unable to create symbolic link sigar library.\n");
        }
    }

    my $oldWarFile="$automateServerDir/webapps/automate-schedule.war";
    if ( -e $oldWarFile) {
        system("rm -f $oldWarFile  > /dev/null 2>&1");
    }

    if (defined($javaInstallFile)) {
        # copy the java installer into the downloads dir to have handy in case
        #    an agent should need it too.
        system("cp \"$javaInstallFile\" \"$automateServerDir/webapps/automate-schedule/download\" > /dev/null 2> /dev/null");
    }

    if (defined($restoreCommand)) {
        chdir($automateServerDir);
        system("$restoreCommand") == 0 || 
            die("Unable to restore existing config files.\n");
        system("rm -f existingConfig.tar");
        chdir($startingDir);
    }

    system("echo Removing reference to ServerLifecycleListener for Tomcat 7.");
    system("mv \"$automateServerDir/conf/server.xml\" \"$automateServerDir/conf/server.xml.install.backup\"");
    system("grep -v \"org.apache.catalina.mbeans.ServerLifecycleListener\" \"$automateServerDir/conf/server.xml.install.backup\" > \"$automateServerDir/conf/server.xml\" 2> /dev/null");

    # ============================================================================================
    # Bug 13486 - replace [sslProtocol="TLS"] with [sslEnabledProtocols="TLSv1,TLSv1.1,TLSv1.2"]
    # ============================================================================================
    print("Scanning conf/server.xml for TLS protocol\n");
    my $serverxml = "$automateServerDir/conf/server.xml";
    my $serverxmlbak = $serverxml . '.tls.bak';
    system("rm -f $serverxmlbak");
    system("mv $serverxml $serverxmlbak") == 0 || die("Unable to rename $serverxml to $serverxmlbak\n");
    my $searchText = "sslProtocol=\\\"TLS\\\"";
    my $replaceText = "sslEnabledProtocols=\\\"TLSv1,TLSv1.1,TLSv1.2\\\"";
    ####  print("Scanning conf/server.xml file for $searchText replace it with $replaceText\n");
    $installer->copyScript($serverxmlbak, $serverxml, $searchText, $replaceText) or die("Error scanning server.xml for TLS protocol\n");
    system("rm -f $serverxmlbak");

    # Bug 3267 and 3407
    system("rm -f $automateServerDir/webapps/automate-schedule/WEB-INF/db/audit_translations/agent_event_file.yml > /dev/null 2>&1");    
}

sub checkIfServerRunning {
    my $file= "$automateServerDir/server.pid";
    if ( -r $file ) {
       my $OLD_PID=`cat ${file}`;
       chomp($OLD_PID);
       my $PROCESS_COUNT=`ps -p ${OLD_PID} | grep ${OLD_PID} | wc -l`;
       if ( ${PROCESS_COUNT} > 0 ) {
            die("Server process $OLD_PID in $automateServerDir needs to be shutdown before installation can continue.\n");
        }
    }
}

sub checkIfOldServerRunning {
    my $file= "$oldSkybotServerDir/server.pid";
    if ( -r $file ) {
       my $OLD_PID=`cat ${file}`;
       chomp($OLD_PID);
       my $PROCESS_COUNT=`ps -p ${OLD_PID} | grep ${OLD_PID} | wc -l`;
       if ( ${PROCESS_COUNT} > 0 ) {
            die("Server process $OLD_PID in $oldSkybotServerDir needs to be shutdown before installation can continue.\n");
        }
    }
}

sub manualConfig {
    $httpPort = $installer->promptForPort("Enter the port number to use for web browsers", $httpPort);
    $tlPort = $installer->promptForPort("Enter the port number to use for agent connections", $tlPort);
    $ajpPort = $installer->promptForPort("Enter the port number to use for AJP connections", $ajpPort);
    $shutdownPort = $installer->promptForPort("Enter the port number to use for shutdown notification", $shutdownPort);
    $dbPort = $installer->promptForPort("Enter the port number to use for the database server", $dbPort);
}

sub downloadAgents {
    print("Checking if the Agent Installers are up to date...\n");
    # fetch the agent_checksum file
    my $result = $installer->forceDownload('agent_checksum');
    if ($result == 0) {
        print("Unable to auto-download the Automate Schedule Agent installer information\n");
        print("You will need to manually download the Agent installers from our website.\n");
        return 0;
    }

    my $downloadDir = "$automateServerDir/webapps/automate-schedule/download";
    if (! -d $downloadDir) {
        $result = $installer->makeDir($downloadDir);
        if ($result == 0) {
            print("Unable to create the Agent Installer download directory\n");
            print("   $downloadDir\n");
            return 0;
        }
    }
    my $checksums = $installer->readFileHash('agent_checksum');
    while ((my $key, my $value) = each %$checksums) {

        my $needToDownloadIt = 0;
        if (-e "$downloadDir/$key") {
            my $line = `sum -r $downloadDir/$key`;
            if ($? != 0) {
                print("Unable to determine checksum of the existing agent installers.\n");
                print("You will need to manually determine if a newer Agent installer is present.\n");
                return 1;
            }
            chomp($line);
            my ($checksum, $ignored) = split / /, $line;
            if ($checksum == $value) {
                print("The download file $key is up to date.\n");
                next;
            } else {
                print("A newer version of $key is available.\n");
                $needToDownloadIt = 1;
            }
        } else {
            $needToDownloadIt = 1;
        }

        if ($needToDownloadIt) {
            # see if the file exists in the current dir
            if (-e $key) {
                $needToDownloadIt = 0;
                my $line = `sum -r $key`;
                chomp($line);
                my ($checksum, $ignored) = split / /, $line;
                if ($checksum == $value) {
                    print("The local file $key is up to date.\n");
                } else {
                    print("The local file $key is not current,\n");
                    print("  but will be used anyway to install/update an Agent on this system.\n");
                }
            }
        }

        if ($needToDownloadIt) {
            $result = $installer->forceDownload($key);
            if ($result == 0) {
                print("You will need to manually download the Agent Installers from our website.\n");
                return 0;
            }
        }
        
        $result = system("cp $key $downloadDir");
        my $failure = $!;
        if ($result != 0) {
            print("Unable to copy agent installers to $downloadDir. $failure\n");
            return 0;
        }
        $needToInstallAgent = 1;
    }

    return 1;
}

sub copyAgents {
    my $downloadDir = "$automateServerDir/webapps/automate-schedule/download";
    my $result;
    if (! -d $downloadDir) {
        $result = $installer->makeDir($downloadDir);
        if ($result == 0) {
            print("Unable to create the Agent Installer download directory\n");
            print("   $downloadDir\n");
            return 0;
        }
    }

    $result = system("cp setupAutomateScheduleAgent.* $downloadDir;cp setupAutomateScheduleIBMiAgent.exe $downloadDir");
    my $failure = $!;
    if ($result != 0) {
        print("Unable to copy agent installers to $downloadDir. $failure\n");
        return 0;
    }
    return 1;
}
# Clean up the installed file's attributes and Finish

if ($whoami eq 'root') {
    # change ownership of everything over to automate.
    my $search = new File::List("$automateServerDir");
    my @files = @{ $search->find(".") };
    foreach my $file (@files) {
        my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
            $atime,$mtime,$ctime,$blksize,$blocks) = stat($file);
        if ($uid eq 0) {
            system("chown $automateUser:$skybotGID \"$file\"") == 0 || 
                die("Unable to change ownership of $file to $automateUser\n");
        }
    }

    $search = new File::List("$automateServerDir");
    $search->show_empty_dirs();
    $search->show_only_dirs();
    my @dirs = @{ $search->find(".") };
    foreach my $dir (@dirs) {
        my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
            $atime,$mtime,$ctime,$blksize,$blocks) = stat($dir);
        if ($uid eq 0) {
            system("chown $automateUser:$skybotGID \"$dir\"") == 0 || 
                die("Unable to change ownership of $dir to $automateUser\n");
        }
    }
    
}

# If this is a rebrand
if ($rebrand == 1) {

    # Remove these files:
    system("rm -f $automateServerDir/startskybotdb.sh > /dev/null 2>&1");
    system("rm -f $automateServerDir/stopskybotdb.sh > /dev/null 2>&1");
    system("rm -f $automateServerDir/SkybotLicenseAgreement.html > /dev/null 2>&1");
    system("rm -f $automateServerDir/skybot-scheduler.mib > /dev/null 2>&1");
    system("rm -f $automateServerDir/skybotSSL.jks > /dev/null 2>&1");
    
    # and add this symlink:
    # Do not do this if the Agent has already.
    if ( -d $oldSkybotDir) {
        print("* Symlink already exists for $oldSkybotDir\n");
    } else {
        print("* Making symlink to $automateScheduleDir for $oldSkybotDir\n");
        system("ln -sf $automateScheduleDir $oldSkybotDir > /dev/null 2>&1");
    }
}

# Make sure everything is readable and executable
system("chmod 755 $automateServerDir/*.sh") == 0 || 
    die("Unable to set the file-executable rights on $automateServerDir/*.sh: $!\n");
system("chmod -R 755 $automateServerDir/lib") == 0 || 
    die("Unable to set the file-executable rights on $automateServerDir/lib: $!\n");
system("chmod -R 755 $automateServerDir/bin") == 0 || 
    die("Unable to set the file-executable rights on $automateServerDir/bin: $!\n");

# resetadmin.sh ships with limited authority
system("chmod 600 $automateServerDir/resetadmin.sh");

my $command = "$automateServerDir/startscheduler.sh";
if ( $whoami eq 'root' ) {
    $command = "su $automateUser -c \"$command\"";
}

print("Server installation has completed.\n");
print("Do you want to start the server now? [y] ");
my $startAgent = 1;
my $done = 0;
while(! $done) {
    my $input = <STDIN>;
    chomp($input);
    if ($input eq '' || $input eq 'y' || $input eq 'Y') {
        print("Starting server with command: $command\n");
        system("$command > $tmpUserLog 2>&1") == 0 || die("Unable to start the server\n" . dumpTempLog($tmpUserLog));
        print("The server may need a few minutes to start up before it's ready for use.\n");
        $done = 1;
    } elsif ($input eq 'n' || $input eq 'N') {
        print("When you're ready to start the server, use the command:\n");
        print("   $command\n");
        $done = 1;
        $startAgent = 0;
    } else {
        print("Please answer Y or N: [y] ");
    }
}


if ($whoami eq 'root') {
    print("If you want the server to start when this system boots up, add this command\n   to this system's startup sequence.\n");
}


my $hostname=`hostname | cut -f1 -d'.'`;
chomp($hostname);
print("Use the link $protocol://$hostname:$httpPort to access the Automate Schedule\n");

if (! -d "$automateScheduleDir/agent") {
    $needToInstallAgent = 1;
}

if ($needToInstallAgent == 0) {
    # all done
    exit(0);
}

# create a local agent-info file
open (DATAFILE, ">local") || die("Unable to create an agent info config file: $!\nA local agent will have to be installed manually.\n");
print DATAFILE "JavaHome=$javaDir\n";
print DATAFILE "InstallDirectory=$automateScheduleDir\n";
my $localPort = $installer->findOpenPort(7471);
print DATAFILE "LocalListenPort=$localPort\n";
print DATAFILE "ServerListenPort=$tlPort\n";
close(DATAFILE);

my $what = 'install';
if (-d "$automateScheduleDir/agent") {
    $what = 'update';
}

my $command = "cd ..;tar xf automatescheduleinstall/setupAutomateScheduleAgent.tar; ./automatescheduleinstall/agentInstall local://localhost local -q";
if ($startAgent == 0) {
    $command = $command . " -nostart";
}
exec($command) == 0 || die("Unable to $what a local agent: $!\n");

sub dumpTempLog {
    my $logFile = shift;

    if ( -e $logFile ) {
        system("cat $logFile");
    }

    return '';
}

